#!/local/bin/perl -- -*-perl-*-

# Author: Jeff Dalton

# This program does "most of the work" when you want to change the
# outer package name of a tree of Java files.  It makes a copy of
# the tree under the new outer name, changing package and import
# statements as it goes.

# Syntax: change-package from_dir to_dir old_prefix new_prefix

# to_dir must be absolute, just because File::Find() does a chdir
# to the various subdirectories of from_dir.  /\/

# For example if we are using "oplan" as the outermost package
# and want to change this to "ix", we could do it as follows,
# assuming the current directory contains the "oplan" directory
# that corresponds to the package:

# bin/change-package.pl oplan /tmp/ix oplan ix

# This makes a copy of the "oplan" file tree in /tmp/ix, and in the
# copied .java files "package oplan..." and "import oplan..." will
# be changed to "package ix..." and "import ix...".

# Only .java files are copied.

use File::Find;

$debug = 0;

$mkdir_mode = oct "755";	# drwxr-xr-x

if ($#ARGV != 3)
{
    print "usage: change-package from_dir to_dir old_prefix new_prefix\n";
    exit 1;
}

($from_dir, $to_dir, $old_prefix, $new_prefix) = @ARGV;

if (substr($to_dir,0,1) ne "/") {
    print "usage: to_dir $to_dir is not absolute.\n";
    exit 1;
}
print "Copying from $from_dir to $to_dir\n";
print "Changing packages from $old_prefix to $new_prefix\n";

find(\&package_changer, $from_dir);

sub package_changer {
    # N.B. when $name is "." $fullname does not end "/."

    my $fullname = $File::Find::name;
    my $dir = $File::Find::dir;
    my $name = $_;

    my $from_suffix = substr($fullname, (length $from_dir));

    my $to_fullname = $to_dir . $from_suffix;

    if (-d $name) {
	# A directory.
	if ($debug) {print "----- Directory: ", $fullname, " -----\n";}
	(not -e $to_fullname)
	    or die "Directory $to_fullname already exists\n";
	if ($debug) {print "mkdir $to_fullname\n";}
	mkdir $to_fullname, $mkdir_mode
	    or die "Cannot mkdir $to_fullname\n";
    }
    elsif (substr($name, -5) eq ".java") {
	# .java file
	(not -e $to_fullname)
	    or die "File $to_fullname already exists\n";
	open IN, "<$name"
	    or die "Can't open input file $fullname\n";
	open OUT, ">$to_fullname"
	    or die "Can't open output file $to_fullname\n";
	my $change_p = "";
	while (<IN>) {
	    if (m/^(package|import) /) {
		# A package or import statement.
		my $statement = $1;
		my $package = $'; 		# really the rest of the line
		if ($package =~ m/^$old_prefix/o) {
		    my $suffix = $';
		    if ($debug) {
			unless ($change_p) {
			    print $fullname, " --> ", "$to_fullname\n";
			}
			$change_p = 1;
			print "< $statement $package";
			print "> $statement $new_prefix$suffix";
		    }
		    print OUT "$statement $new_prefix$suffix";
		}
		else {
		    # package or import does not use $old_prefix
		    print OUT $_;
		}
	    }
	    else {
		# Not a package or import statement.
		if (m/$old_prefix/o) {
		    print "Found $old_prefix in file $fullname\n",
		          "line : ", $_;
		}
		if (m/\@see $old_prefix/o) {
		    s/\@see $old_prefix/\@see $new_prefix/o;
		    print "Changing it to:\n$_\n";
		}
		print OUT $_;
	    }
	}
	close IN;
	close OUT;
    }
    elsif (substr($name, -6) eq ".class") {
	# Skip .class files
    }
    else {
	# Tell user about files that aren't .class or .java
	print "Skipping file $fullname\n";
    }

}
