package ixagentwizard;

import java.awt.*;
import java.io.*;

import javax.swing.*;

/**
 * <p>Title: I-X Agent Wizard</p>
 *
 * <p>Description: Tool for managing I-X agents/panels</p>
 *
 * <p>This class represents the application that is the I-X Agent Wizard.
 * Running this class will bring up a window that allows the simple creation of
 * an I-X Agent or Process Panel. </p>
 *
 * <p>Copyright: Copyright (c) 2004</p>
 *
 * <p>Company: University of Edinburgh</p>
 *
 * @author Gerhard Wickler
 * @version 1.0
 */
public class IXAgentWizardClass {

    // the frame containing GUI of this application:
    protected IXAgentWizardFrame frame;
    // a FileChooser for selecting directories
    protected JFileChooser dirSelect;
    // a FileChooser for selecting files
    protected JFileChooser fileSelect;
    // a MessageBox for displaying errors:
    protected ErrorMessageBox errorBox;
    // the base directory in which I-X is installed:
    protected String ixBaseDir = null;

    /**
     * <p>This constructor creates and shows the application. It constructs an
     * instance of an <code>IXAgentWizardFrame</code>, makes sure it fits on the
     * screen, and centers it before displaying it. Finally, the logic side of
     * the application is initialized by calling <code>initLogic()</code>. </p>
     */
    public IXAgentWizardClass() {
        frame = new IXAgentWizardFrame(this);
        //frame.pack();
        frame.validate();

        // make sure it fits and center the window:
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        Dimension frameSize = frame.getSize();
        if (frameSize.height > screenSize.height) {
            frameSize.height = screenSize.height;
        }
        if (frameSize.width > screenSize.width) {
            frameSize.width = screenSize.width;
        }
        frame.setLocation((screenSize.width - frameSize.width) / 2,
                          (screenSize.height - frameSize.height) / 2);
        frame.setVisible(true);
        try {
            jbInit();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /**
     * <p>This function initializes the logical side of the application. First
     * it attempts to find the I-X base directory, that is, the directory in
     * which I-X is installed. If the parent directory of the directory in which
     * the agent wizard is run begins with "ix" or "i-x" (case-insensitive) this
     * is assumed to be the I-X base directory. The agent directory is initially
     * set to the "apps" directory within the I-X base directory, if it exists.
     * Fianlly, the default icon path is set to the current directory. </p>
     *
     * @param frame IXAgentWizardFrame the GUI for this application
     */
    private void jbInit() {
        // initialize the directory chooser:
        dirSelect = new JFileChooser();
        dirSelect.setMultiSelectionEnabled(false);
        dirSelect.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        // initialize the file chooser:
        fileSelect = new JFileChooser();
        fileSelect.setMultiSelectionEnabled(false);
        fileSelect.setFileSelectionMode(JFileChooser.FILES_ONLY);
        // initialize MessageBox for displaying errors:
        errorBox = new ErrorMessageBox(frame, "Problem", true);
        // try to find the I-X base directory:
		try {
	        File parentDir = new File("..");
			String path = parentDir.getCanonicalPath();
	        int lastSepPos = path.lastIndexOf((int) File.separatorChar);
	        String ixDir = path.substring(lastSepPos + 1, path.length()).toLowerCase();
	        if (ixDir.startsWith("ix") || ixDir.startsWith("i-x")) {
	            ixBaseDir = path;
	        }
		} catch (IOException e) {
			e.printStackTrace();
		}
        // or get it from the user if necessary:
        while (ixBaseDir == null) {
            dirSelect.setDialogTitle("Select the I-X base directory");
            if (dirSelect.showDialog(frame, "Select") == JFileChooser.APPROVE_OPTION) {
                ixBaseDir = dirSelect.getSelectedFile().getPath();
            } else {
                displayErrorMessage(
                        "You must specify the I-X base directory!\n" +
                        "This is the directory in which I-X is installed, e.g. \"ix-4.5\".");
            }
        }
        frame.statusBar.setText("I-X base directory: " + ixBaseDir);
        // set the agent directory to the apps directory, if it exists:
        String appsDirName = ixBaseDir + File.separatorChar + "apps";
        File appsDir = new File(appsDirName);
        if (appsDir.isDirectory()) {
            frame.agentDirPathTextField.setText(appsDirName + File.separatorChar + "newapp");
        } else {
            frame.agentDirPathTextField.setText("");
        }
        // set the icon directory:
        File currentDir = new File("./logo.gif");
        frame.iconPathTextField.setText("");
		try {
			String path = currentDir.getCanonicalPath();
	        frame.iconPathTextField.setText(path);
		} catch (IOException e) {
			e.printStackTrace();
		}
    }

    public boolean createAgent(String agentDir, String dName, String sName) {
        try {
            char sep = File.separatorChar;
            agentDir += sep ;//+ sName + sep;

            File newDir = new File(agentDir + "config");
            newDir.mkdirs();
            newDir = new File(agentDir + "domain-library");
            newDir.mkdirs();
            newDir = new File(agentDir + "images");
            newDir.mkdirs();
            newDir = new File(agentDir + "java");
            newDir.mkdirs();
            newDir = new File(agentDir + "resources");
            newDir.mkdirs();
            newDir = new File(agentDir + "scripts" + sep + "unix");
            newDir.mkdirs();
            newDir = new File(agentDir + "scripts" + sep + "win");
            newDir.mkdirs();

            // write the windows script:
            FileWriter fw = new FileWriter(
                    agentDir + "scripts" + sep + "win" + sep + sName + ".bat");
            fw.write(":: file created by the I-X Agent Wizard\n" +
                     "@echo off\n" +
                     "title " + dName + " Command Console\n");
            fw.write(":: cd to the I-X application base\n" +
                     "cd ../..\n" +
                     "set app_base=.\n" +
                     "set app_path=./java\n");
            fw.write("set app_params=-ipc=xml" +
                     " -load config/" + sName + "-panel.props" +
                     " -plan-state-to-save=*");
            fw.write((frame.serverCheckBox.isSelected()) ?
                     " -run-name-server\n" : "\n");
            fw.write("set ix_base=../..\n" +
                     "set java_params=\n" +
                     "set ipc=xml\n" +
                     "%ix_base%/scripts/win/run-ix-app.bat ix.ip2.Ip2\n" +
                     "pause" + '\n');
            fw.close();

            // write the properties file:
            fw = new FileWriter(
                    agentDir + "config" + sep + sName + "-panel.props");
            fw.write("# I-X Process Panel Property File\n\n" +
                     "# Panel naming and appearance\n" +
                     "symbol-name=" + sName + "\n" +
                     "display-name=" + dName + " Panel\n" +
                     "logo-image=images/" + sName + "-logo.gif\n\n");
            fw.write("# Domain model\n" +
                     "domain=domain-library/" + sName +
                     "-process-model.xml\n\n");
            fw.write("#Test menu\n" +
                     "test-menu=config/" + sName + "-test-menu.xml\n\n");
            fw.write("# I-Space\n");
            fw.close();

            // copy the image file:
            File imageFile = new File(frame.iconPathTextField.getText());
            byte[] imageData = new byte[(int) imageFile.length()];
            FileInputStream fis = new FileInputStream(imageFile);
            if (fis.read(imageData) == -1) {
                throw new AssertionError("Unexpected end of image icon file!");
            }
            fis.close();
            FileOutputStream fos = new FileOutputStream(
                    agentDir + "images" + sep + sName + "-logo.gif");
            fos.write(imageData);
            fos.close();

            // write the domain model:
            fw = new FileWriter(agentDir + "domain-library" + sep +
                                sName + "-process-model.xml");
            fw.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
                     "<domain xmlns=\"http://www.aiai.ed.ac.uk/project/ix/\">\n" +
                     " <refinements>\n" +
                     "  <list />\n" +
                     " </refinements>\n" +
                     "</domain>\n");
            fw.close();

            // write the test menu:
            fw = new FileWriter(agentDir + "config" + sep +
                                sName + "-test-menu.xml");
            fw.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
                     "<list>\n\n" +
                     "   <test-item\n" +
                     "         menu-text=\"test menu entry text\"\n" +
                     "         to-name=\"me\">\n" +
                     "      <contents>   \n" +
                     "         <activity priority=\"low\" report-back=\"no\">\n" +
                     "            <pattern>\n" +
                     "               <list>\n" +
                     "                  <symbol>activity</symbol>\n" +
                     "                  <symbol>parameter</symbol>\n" +
                     "               </list>\n" +
                     "            </pattern>\n" +
                     "         </activity>\n" +
                     "      </contents>\n" +
                     "   </test-item>\n\n" +
                     "</list>");
            fw.close();

        } catch (Exception ioe) {
            displayErrorMessage(
                    "Unable to create agent:\n" + ioe.getMessage());
            return false;
        }
        return true;
    }

    /**
     * <p>This function displays a MessageBox informing the user of the given
     * message. </p>
     *
     * @param msg String the problem message to be displayed
     */
    protected void displayErrorMessage(String msg) {
        errorBox.setMessageText(msg);
        Dimension dlgSize = errorBox.getPreferredSize();
        Dimension frmSize = frame.getSize();
        Point loc = frame.getLocation();
        errorBox.setLocation((frmSize.width - dlgSize.width) / 2 + loc.x,
                             (frmSize.height - dlgSize.height) / 2 + loc.y);
        errorBox.pack();
        errorBox.setVisible(true);
    }

    /**
     * <p>This function creates the I-X Agent Wizard application. All it does is
     * set the look and feel to the one used by the system and create an
     * instance of this class. </p>
     *
     * @param args String[] will be ignored
     */
    static public void main(String[] args) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                // set the look and feel:
                try {
                    UIManager.setLookAndFeel(
                            UIManager.getSystemLookAndFeelClassName());
                } catch (Exception exception) {
                    exception.printStackTrace();
                }
                // create the class instance:
                new IXAgentWizardClass();
            }
        });
    }

}
