/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jun  4 13:09:54 2007 by Jeff Dalton
 * Copyright: (c) 1998, 2002 - 2007, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.io.*;
import java.util.*;

import junit.framework.*;

import org.jdom.Element;

import ix.icore.*;
import ix.icore.domain.Domain;
import ix.icore.plan.Plan;
import ix.applet.AppletMessage;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.reflect.ClassSyntax;

/** 
 * Test cases for some XML utilities, primarily {@link XMLTranslator}.
 */

public class XMLTest extends TestCase {

    XMLConfig config = new ix.ip2.Ip2XMLConfig();
    ClassSyntax syntax = config.defaultClassSyntax();

    StructuralEquality strictEquality = new StrictEquality(syntax);
    StructuralEquality flexibleEquality = new FlexibleEquality(syntax);

    public XMLTest(String name) {
	super(name);
    }

    public void testEncodeDecode() {
	assertEquals("<Some&Text>",
		     XML.decodeText(XML.encodeText("<Some&Text>")));
	assertEquals("<&>",
		     XML.decodeText(XML.encodeText("<&>")));
	assertEquals("a<<b",
		     XML.decodeText(XML.encodeText("a<<b")));
	    
    }

    public void testBasicIssueConversion() {
	Issue from = new Issue("do something");
	Issue to = (Issue)XML.objectFromXML(XML.objectToXMLString(from));
	assertEquals(from.getVerb(), to.getVerb());
	assertEquals(from.getParameters(), to.getParameters());
    }

    public void testBasicReportConversion() {
	Report from = new Report("did something");
	Report to = (Report)XML.objectFromXML(XML.objectToXMLString(from));
	assertEquals(from.getText(), to.getText());
    }

    public void testAppletMessageConversion() {
	// This failed, the command "do it" becomming null due
	// to the classic error:
	//   public void setCommand(String commmand) {
	//       this.command = command;
	//   }
	// Note the extra "m" in the first line.
	AppletMessage from = new AppletMessage("sender", "do it", "now");
	AppletMessage to =
	    (AppletMessage)XML.objectFromXML(XML.objectToXMLString(from));
	assertEquals(from.toList(), to.toList());
    }

    public void testDoubleAsFieldBug() {
	String xml =
	    "<list xmlns=\"http://www.aiai.ed.ac.uk/project/ix/\">" +
            "  <node-position name=\"3\" x=\"216.0\" y=\"142.0\" />" +
            "  <double>3.2</double>" +
	    "</list>";
	XML.objectFromXML(xml);
    }

    public void testPlainTranslator() {
	// A plain XMLTranslator uses impl attributes and so should
	// let us get back objects that are strictly structurally equal.
	XMLTranslator xmlt = new XMLTranslator(syntax);
	Debug.expect(xmlt.omitImplAttributes == false);
	translatorTest(xmlt,
		       strictEquality);
    }

    public void testIp2Translator() {
	// An Ip2 XMLTranslator does not use impl attributes and so
	// we must use the flexible equality test that doesn't care
	// about the actual class of collections or maps.
	XMLTranslator xmlt = config.defaultXMLTranslator();
	translatorTest(xmlt,
		       flexibleEquality);
    }

    protected void translatorTest(XMLTranslator xmlt,
				  StructuralEquality eq) {

	check(xmlt, eq, new Long(123));

	check(xmlt, eq, Lisp.NIL);

	check(xmlt, eq, Lisp.readFromString("(10 11 12)"));

	check(xmlt, eq, 
	      new LinkedList((LList)Lisp.readFromString("(10 11 12)")));

	{ Issue issue = new ix.icore.Issue("Note this");
	  issue.setPriority(ix.icore.Priority.HIGH);
	  issue.setRef(Name.valueOf("ref-1"));
	  issue.setSenderId(Name.valueOf("no one"));
	  check(xmlt, eq, issue);

	  // Try double quote in attribute
	  issue.setSenderId(Name.valueOf("probably \"no one\""));
	  check(xmlt, eq, issue);
	}

	{ List ll = new LinkedList();
	  ll.add("a"); ll.add("b"); ll.add("c");
	  check(xmlt, eq, ll);
	}

	{ Map mm = new HashMap();
	  mm.put("a", "a-val");
	  mm.put("b", "b-val");
	  check(xmlt, eq, mm);
	}

	check(xmlt, eq, "String with <boo f=\"x\">embedded XML &c</boo>");

	check(xmlt, eq, ix.icore.Status.COMPLETE);

	check(xmlt, eq, new ix.ichat.ChatMessage("Hello", "sender-1"));

	check(xmlt, eq, Lisp.readFromString("(a nil b)"));

	check(xmlt, eq, new ix.icore.domain.NodeSpec("node-1", Lisp.NIL));

	// /\/: Can't yet compare JDOM documents for structural equality.

//  	check(xmlt, eq,
//  	      new LiteralDocument
//  	          (new Element("test-element")
//  	               .setAttribute("attrib", "atval")
//  	               .addContent(new Element("test-subelement"))));

	// If we just try new Date(), it usually won't work because
	// translation is only to the second.
	check(xmlt, eq,
	      new GregorianCalendar(2005, 3, 17, 22, 05, 03)
	          .getTime());

	check(xmlt, eq, new Duration("P1D"));

	check(xmlt, eq,
	      Lisp.list(new ix.iview.igraph.NodePosition
			        (Name.valueOf("3"),
				 216.0, 142.0),
			new Double(3.2)));

    }

    protected void check(XMLTranslator xmlt,
			 StructuralEquality eq,
			 Object obj) {
	String xml = xmlt.objectToXMLString(obj);
	// Debug.noteln("XML:", xml);
	Object back = xmlt.objectFromXML(xml);
	if (eq.equal(obj, back) == false)
	    fail("Failed to translate back " + obj +
		 "; instead received " + back);
    }

    public void testDomainTranslation() {
	// This one processes all objects in "test-domain"
	// even though some of them aren't domains.  For example,
	// "standard-tests is a list of plan-tests; and some
	// of the files are initial plans for tests.  The only
	// files we have to skip are ones containing Lisp
	// support code for compute conditions.
	XMLTranslator xmlt = config.defaultXMLTranslator();
	FileSyntaxManager fsm = config.defaultFileSyntaxManager();
	File[] files = fsm.getFiles(Object.class, "test-domains");
	for (int i = 0; i < files.length; i++) {
	    String name = Strings.beforeLast(".", files[i].getName());
	    if (name.endsWith("support"))
		// Skip files that contain Lisp code.
		continue;
	    Object obj = fsm.readObject(Object.class, files[i].getPath());
	    try {
		check(xmlt, flexibleEquality, obj);
	    }
	    catch (Exception e) {
		Debug.out.println("Problem with file " + files[i]);
		Debug.noteException(e);	// make sure we see the exception
		throw new RethrownException(e);
	    }
	}
    }

    public void testPlanTranslation() {
	XMLTranslator xmlt = config.defaultXMLTranslator();
	FileSyntaxManager fsm = config.defaultFileSyntaxManager();
	File[] files = fsm.getFiles(Plan.class, "test-results");
	for (int i = 0; i < files.length; i++) {
	    Object obj = fsm.readObject(Plan.class, files[i].getPath());
	    try {
		check(xmlt, flexibleEquality, obj);
	    }
	    catch (Exception e) {
		Debug.out.println("Problem with file " + files[i]);
		Debug.noteException(e);	// make sure we see the exception
		throw new RethrownException(e);
	    }
	}
    }

    public static class StrictEquality extends StructuralEquality {

	public StrictEquality(ClassSyntax s) { super(s); }

	// Because the result should always be true, we can
	// throw an imformative exception if it would be false
	// rather than just returning false.  This way we
	// can find out which subobject caused the failure.

	public boolean equal(Object a, Object b) {
	    boolean result = super.equal(a, b);
	    if (result == true)
		return result;
	    else
		throw new IllegalArgumentException
		    ("Mismatched objects 1: " + a + " and 2: " + b);
	}

    }

    public static class FlexibleEquality extends StrictEquality {

	public FlexibleEquality(ClassSyntax s) { super(s); }

	protected boolean haveEquivalentClasses(Object a, Object b) {
	    Class c_a = a.getClass();
	    Class c_b = b.getClass();
	    return c_a == c_b
		|| (a instanceof List && b instanceof List)
		|| (a instanceof Set && b instanceof Set)
		|| (a instanceof Map && b instanceof Map);
	}
    }

}
