/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Copyright: (c) 2001, 2008, AIAI, University of Edinburgh
 * Updated: Thu Apr 24 18:29:36 2008 by Jeff Dalton
 */

package ix.util.context;

import junit.framework.*;

/**
 * Test cases for context-holding strategies.
 */
public class ContextHoldingStrategyTest extends TestCase {

    ContextHoldingStrategy originalStrategy
	= Context.getContextHoldingStrategy();

    public ContextHoldingStrategyTest(String name) {
	super(name);
    }

    public void tearDown() {
	Context.setContextHoldingStrategy(originalStrategy);
    }

    public void testGlobalHoldingStrategy() {
	ContextHoldingStrategy s = new GlobalHoldingStrategy();
	ContextHolder h = s.getContextHolder();
	assertSame("holder 1", h, s.getContextHolder());

	Context.setContextHoldingStrategy(s);
	assertSame("strategy", s, Context.getContextHoldingStrategy());
	assertSame("holder 2", h, Context.getContextHolder());
    }

    public void testThreadLocalHoldingStrategy() {
	final ContextHoldingStrategy s = new ThreadLocalHoldingStrategy();
	final ContextHolder h = new BasicContextHolder();

	Context.setContextHoldingStrategy(s);

	Thread t1 = new Thread() {
	    public void run() {
		// Initially, we should get a null holder.
		// /\/: That's changed.
// 		assertNull("null 1", Context.getContextHolder());

		// But we can assign a specific holder
		s.setContextHolder(h);
		assertSame("holder 1", h, Context.getContextHolder());

		// A child thread should still get null
		// /\/: Now the child should get a different holder
		Thread t2 = new Thread() {
		    public void run() {
// 			assertNull("null 2", Context.getContextHolder());
			assertNotSame("t2", h, Context.getContextHolder());
		    }
		};
		t2.start();
		try { t2.join(); } catch (InterruptedException e) {};
	    }
	};

	t1.start();
	try { t1.join(); } catch (InterruptedException e) {};

    }

    public void testInheritableThreadLocalHoldingStrategy() {
	final ContextHoldingStrategy s
	    = new InheritableThreadLocalHoldingStrategy();
	final ContextHolder h = new BasicContextHolder();

	Context.setContextHoldingStrategy(s);

	Thread t1 = new Thread() {
	    public void run() {
		// Initially, we should get a null holder.
		// /\/: That's changed.
// 		assertNull("null 1", Context.getContextHolder());

		// But we can assign a specific holder
		s.setContextHolder(h);
		assertSame("holder 1", h, Context.getContextHolder());

		// A child thread should inherit the parent's holder.
		Thread t2 = new Thread() {
		    public void run() {
			assertSame("child 1", h, Context.getContextHolder());

			// Now try changing it.
			ContextHolder h2 = new BasicContextHolder();
			s.setContextHolder(h2);
			assertSame("child 2", h2, Context.getContextHolder());

		    }
		};
		t2.start();
		try { t2.join(); } catch (InterruptedException e) {};

		// Change in child should not have changed parent's holder
		assertSame("holder 2", h, Context.getContextHolder());
	    }
	};

	t1.start();
	try { t1.join(); } catch (InterruptedException e) {};

    }

}
