/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue May 13 16:48:47 2008 by Jeff Dalton
 * Copyright: (c) 2004, 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import junit.framework.*;

import java.util.*;
import java.util.concurrent.*;

import ix.iplan.*;
import ix.icore.domain.Domain;
import ix.icore.plan.Plan;
import ix.icore.plan.build.*;

import ix.util.*;
import ix.util.context.*;
import ix.util.lisp.*;
import ix.util.xml.*;

import ix.util.reflect.ClassDescr;
import ix.util.reflect.FieldDescr;

/** Simple multi-thread planning test. */

public class SimpleMultiThreadTest extends TestCase {

    String domainName = "ix/iplan/domain-library/paint-1.lsp";

    String planName = "ix/iplan/expected-plans/paint-1-test1-plan-1.xml";

    public SimpleMultiThreadTest(String name) {
	super(name);
	Debug.noteThreads = true; // /\/
    }

    public void testSeveralPlanners() {
	Context.setContextHoldingStrategy(new ThreadLocalHoldingStrategy());
	runNPlanners(3);
    }

    void runNPlanners(int n) {
	CountDownLatch finished = new CountDownLatch(n);
	for (int i = 0; i < n; i++) {
	    planInNewThread(finished);
	}
	try {
	    finished.await();	// wait for all to finish
	    Debug.noteln("All planners finished");
	}
	catch (InterruptedException e) {
	    Debug.noteln("runNPlanners interrupted");
	}
    }

    void planInNewThread(CountDownLatch finished) {
	new PlanningThread(finished).start();
    }

    class PlanningThread extends Thread {

	CountDownLatch finished;

	PlanningThread(CountDownLatch finished) {
	    this.finished = finished;
	}

	public void run() {
	    try {
		Debug.noteln("Making planner");
		Planner planner = IPlan.makeDefaultPlanner();
		Domain dom = XML.readObject(Domain.class, domainName);
		planner.setDomain(dom);
		planner.loadPlan(initialPlan("test1"));
		planner.plan();
		Plan expected = XML.readObject(Plan.class, planName);
		Plan found = planner.getPlan();
		assertTrue(new Equality().equal(expected, found));
		Debug.noteln("Plan looks ok.");
	    }
	    finally {
		finished.countDown();
	    }
	}

    }

    Plan initialPlan(String task) {
	PlanBuilder builder = new SimplePlanBuilder();
	LList taskPattern = Lisp.elementsFromString(task);
	builder.addActivity(new ix.icore.Activity(taskPattern));
	return builder.getPlan();
    }

    static class Equality extends XMLTest.FlexibleEquality {

	Equality() {
	    super(XML.config().defaultClassSyntax());
	}

	public boolean equal(Object a, Object b) {
	    boolean result = super.equal(a, b);
	    if (result == false) {
		throw new RuntimeException
		    (a + " does not equal " + b);
	    }
	    return result;
	}

	protected boolean missingFieldIsOk(ClassDescr cd, FieldDescr fd,
					   Object f_a, Object f_b) {
	    throw new RuntimeException
		("Missing field " + fd.getName() +
		 " in " + (f_a == null ? "a" : "b"));
	}

    }

}
