/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Nov 15 18:33:29 2004 by Jeff Dalton
 * Copyright 2002, 2003, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.io.PrintStream;
import java.util.*;

import ix.iface.domain.LTF_Parser; // help with constraint syntax /\/
import ix.util.match.*;		   // for LTF_Parser constraint parsers /\/

import ix.util.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

/**
 * Base for objcts that describe the (XML) syntax of data objects
 * in various forms.
 */
public class XMLSyntax {

    protected XMLTranslator xmlt;
    protected ClassSyntax classSyntax;

    protected InheritanceTree inheritance; // unfortunate global /\/

    public XMLSyntax() {
	this(XML.defaultTranslator());
    }

    public XMLSyntax(XMLTranslator xmlt) {
	super();
	this.xmlt = xmlt;
	this.classSyntax = xmlt.getClassSyntax();
//  	Debug.noteln("Constraint syntax:",
//  		     XML.objectToXMLString(getConstraintSyntaxList()));
    }

    /*
     * Useful mappings.
     */

    protected ClassDescr getClassDescr(Class c) {
	return classSyntax.getClassDescr(c);
    }

    protected String getNTName(ClassDescr cd) {
	return cd.getUpperName();
    }

    protected String getNTName(Class c) {
	return getNTName(getClassDescr(c));
    }

    protected String getUpperName(ClassDescr cd) {
	return cd.getUpperName();
    }

    protected String getUpperName(Class c) {
	return getUpperName(getClassDescr(c));
    }

    protected String getElementName(ClassDescr cd) {
	return cd.getExternalName();
    }

    protected String getElementName(Class c) {
	return getElementName(getClassDescr(c));
    }

    protected String getElementName(FieldDescr fd) {
	return fd.getExternalName();
    }

    /*
     * Building lists of relevant classes.
     */

    public List relevantClasses(Class rootClass) {
	return XML.config().xmlSyntaxClasses(classSyntax, rootClass);
    }

    /*
     * Class utilities
     */

    /** 
     * Adds all classes C that need list-of C definitions.
     * 
     * @param classes  candidate classes
     * @param collection  cantains the result
     */
    protected void collectListofClasses(List classes,
					final Collection collection) {
	walkStructFields(classes, new FieldVisitor() {
	    public void visitField(ClassDescr cd, FieldDescr fd) {
		ClassDescr valueDescr = fd.getTypeDescr();
		ClassDescr eltDescr = valueDescr.getEltType();
		if (valueDescr.isList() && eltDescr != null)
		    collection.add(eltDescr.getDescribedClass());
	    }
	});
    }

    protected void walkStructFields(List classes, FieldVisitor fv) {
	for (Iterator ci = classes.iterator(); ci.hasNext();) {
	    Class c = (Class)ci.next();
	    ClassDescr cd = getClassDescr(c);
	    if (cd.isStruct()) {
		cd.visitFields(fv);
	    }
	}
    }

    /*
     * Field categories
     */

    protected List attributeFields(List fields) {
	return (List)Collect.filter(fields, attributeFieldP);
    }

    protected List elementFields(List fields) {
	return (List)Collect.filter(fields, Fn.negate(attributeFieldP));
    }

    protected Predicate1 attributeFieldP = new Predicate1() {
        public boolean trueOf(Object obj) {
	    FieldDescr fd = (FieldDescr)obj;
	    return xmlt.isAttributeClass(fd.getType());
	}
    };


    /**
     * Returns a list of constraint objects that represent the
     * currently supported syntactic possibilities for constaints.
     */
    protected List getConstraintSyntaxList() {
	final LTF_Parser ltfp = new LTF_Parser();
	List parsers = ltfp.getConstraintParsers().getCases();
	return (List)Collect.map(new LinkedList(), parsers, new Function1() {
	    public Object funcall(Object arg) {
		LTF_Parser.ConstraintParser parser =
		    (LTF_Parser.ConstraintParser)arg;
		return parser.makeTemplate();
	    }
	});
    }


    /**
     * Returns a list of the allowed values of an enumeration class.
     *
     * @throws IllegalArgumentException if the class is not
     *   a subclass of {@link ix.util.EnumeratedValue}.
     *
     * @see ix.util.EnumeratedValue
     */
    protected List getEnumerationValues(Class c) {
	if (getClassDescr(c).isEnumeration())
	    return (List)Fn.applyStatic(c, "values", new Object[]{});
	else
	    throw new IllegalArgumentException(c + " is not an enumeration.");
    }


    /**
     * Test loop that repeatedly asks the user for a class name
     * and prints a description of the syntax for objects of
     * that class.
     */
    public static void main(String[] argv) {
	XMLSyntax syntax = new XMLSyntax();
	for (;;) {
	    String in = Util.askLine("Class name:");
	    if (in.equals("bye"))
		return;
	    syntax.describeClass(in, System.out);
	    System.out.println("");
	}
    }

    public void describeClass(String className, PrintStream out) {
	Class c = classSyntax.classForExternalName(className);
	if (c == null) {
	    out.println("Can't find class named " + className);
	    return;
	}
	out.println("Syntax for " + className + ":");
	ClassDescr cd = getClassDescr(c);
	if (cd.isStruct())
	    out.println(structTextDescription(cd));
	else
	    out.println("No description available.");
    }

    public String structTextDescription(ClassDescr cd) {
	// Get fields with attribute fields first.
	List fields = cd.getFieldDescrs();
	List attrFields = attributeFields(fields);
	List eltFields = elementFields(fields);
	List allFields = Collect.append(attrFields, eltFields);
	Debug.expect(fields.size() == allFields.size());

	List lines = new LinkedList();
	for (Iterator fi = allFields.iterator(); fi.hasNext();) {
	    FieldDescr fd = (FieldDescr)fi.next();
	    String kind = xmlt.isAttributeClass(fd.getType())
		? "attribute" : "element";
	    lines.add("   " + kind + " " +
		      fd.getExternalName() + ": " +
		      fd.getTypeDescr().description());
	}
	return Strings.joinLines(lines);
    }

}
