/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jan 31 03:30:28 2005 by Jeff Dalton
 * Copyright: (c) 2003 - 2005, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.io.*;
import java.net.URL;
import java.util.*;

import org.jdom.Document;
import org.jdom.output.XMLOutputter;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Reads and writes object representations.
 */
public interface FileSyntax extends Loader, Saver {

    /**
     * Returns a set of strings representing file types / extensions.
     */
    public List getFileTypes();

    /**
     * Returns a short description of the type of files
     * understood by this FileSyntax.  The result should be
     * a plural phrase such as "XML files" or "domain descriptions".
     */
    public String getFileTypeDescription();

    /**
     * Indicates whether this syntax can be used.  Some syntaxes rely
     * on classes that are not always present in the class-path.
     */
    public boolean isAvailable();

    /**
     * Is reading supported?
     */
    public boolean canRead();

    /**
     * Is writing supported?
     */
    public boolean canWrite();

    /**
     * The classes whose instances can be read in.  General-purpose
     * syntaxes can read (at least) instances of every class that
     * has a {@link ix.util.reflect.ClassSyntax}.  They can return
     * a list containing only Object.class.
     */
    public List readableClasses();

    /**
     * The classes whose instances can be written out.  General-purpose
     * syntaxes can write (at least) instances of every class that
     * has a {@link ix.util.reflect.ClassSyntax}.  They can return
     * a list containing only Object.class.
     */
    public List writableClasses();

    /*
     * Some syntaxes.
     */

    /** XML file syntax */
    public static class XMLFiles extends AbstractFileSyntax {
	private List types = Lisp.list("xml");
	public XMLFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "XML files";
	}
	public boolean canRead() {
	    return true;
	}
	public boolean canWrite() {
	    return true;
	}
	public Object readObject(URL url) throws IOException {
	    return XML.defaultTranslator().readObject(url);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    XML.defaultTranslator().writeObject(obj, file);
	}
    }

    /** RDF file syntax. */
    public static class RDFFiles extends AbstractFileSyntax {
	private List types = Lisp.list("rdf", "owl");
	public RDFFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "RDF and OWL files";
	}
	public boolean isAvailable() {
	    return existsClass("ix.util.rdf.RdfTranslator")
		&& existsClass("com.hp.hpl.jena.rdf.model.Model");
	}
	public boolean canRead() {
	    return true;
	}
	public boolean canWrite() {
	    return true;
	}
	public Object readObject(URL url) throws IOException {
	    Loader loader = (Loader)getTranslator();
	    return loader.readObject(url);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    Saver saver = (Saver)getTranslator();
	    saver.writeObject(obj, file);
	}
	protected Object getTranslator() {
	    return makeInstance("ix.util.rdf.RdfTranslator");
	}
    }

}
