/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 18 17:11:33 2009 by Jeff Dalton
 * Copyright: (c) 2002 - 2006, 2009, AIAI, University of Edinburgh
 */

package ix.util.reflect;

import java.lang.reflect.*;
import java.lang.annotation.*;
import java.util.*;

import java.io.PrintStream;	// for testing

import java.awt.Color;		// to define a stringer

import java.net.URI;		// to define a stringer
import java.net.URISyntaxException;

import java.net.URL;		// to define a stringer
import java.net.MalformedURLException;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.plan.*;

import ix.util.*;
import ix.util.xml.XML;
import ix.util.xml.XMLSchemaSyntax;
import ix.util.lisp.*;

/**
 * Provides a syntax for objects by acting as a factory for class and
 * field descriptions.
 *
 * <p>For a class with fields, ClassSyntax provides the information
 * needed to process instances field-by-field.  It also embodies
 * decisions about which fields should be accessible in this way,
 * how their values are obtained or set, and the order in which
 * they should be visited.
 *
 * <p>It also specifies how low-level objects are represented
 * as Strings and which XML Schema datatype they should have.
 *
 * <p>A {@link ClassFinder} is used to map both ways between
 * external names and classes and to map between Java and external
 * names for classes and fields.
 *
 * @see ClassDescr
 * @see FieldDescr
 * @see ClassFinder
 */

public class ClassSyntax {

    protected ClassFinder classFinder;

    protected Map<Class,ClassDescr> classToDescrCache =
	new HashMap<Class,ClassDescr>();

    protected Map <Class,Stringer> classToStringerMap = 
	new HashMap<Class,Stringer>();

    public ClassSyntax() {
	this(XML.config().defaultClassFinder());
    }

    public ClassSyntax(ClassFinder finder) {
	Debug.expect(finder != null, "Null ClassFinder");
	this.classFinder = finder;
	initStringConversions();
    }

    public ClassFinder getClassFinder() {
	return classFinder;
    }

    /*
     * External Names
     */

    // /\/: Maybe some of these should not be public so that things
    // have to go via the ClassDescr.

    public String externalNameForClass(Class c) {
	return classFinder.nameForClass(c);
    }

    public Class classForExternalName(String externalName) {
	return classFinder.classForName(externalName);
    }

    public String externalNameForField(String javaName) {
	return classFinder.externalFieldName(javaName);
    }

    public String upperNameForClass(Class c) {
	// /\/: If the external names are the same as the Java names,
	// the upper name loses too much information.  For example,
	// NodeEndRef becomes NODEENDREF.
	String externalName = externalNameForClass(c);
	return externalName.toUpperCase();
    }


    /*
     * Making ClassDescrs
     */

    // /\/: Note that only the one-argument case is made unique.
    // It's possible that we should have a separate class called
    // ValueDescr for field values so that only 1-arg ClassDescrs
    // are used.

    public ClassDescr getClassDescr(Class c) {
	ClassDescr cd = classToDescrCache.get(c);
	if (cd == null) {
	    cd = makeClassDescr(c);
	    classToDescrCache.put(c, cd);
	}
	// Debug.noteln(c + " ---> " + cd);
	return cd;
    }

    protected ClassDescr makeClassDescr(Class c) {
	if (TypedList.class.isAssignableFrom(c))
	    return new ClassDescr(this, c, ListOf.elementClass(c));
	else
	    return new ClassDescr(this, c);
    }

    public ClassDescr makeClassDescr(Class collectionClass,
				     Class eltClass) {
	return new ClassDescr(this, collectionClass, eltClass);
    }

    public ClassDescr makeClassDescr(Class mapClass,
				     Class keyClass, Class valClass) {
	return new ClassDescr(this, mapClass, keyClass, valClass);
    }


    /*
     * String representations
     */

    public String xmlSchemaDatatype(Class c) {
	// First try the Stringer, if one exists.
	Stringer s = getStringer(c);
	if (s != null) {
	    String result = s.xmlSchemaDatatype();
	    if (result != null)
		return result;
	}
	// Try the built-in mapping.
	return XMLSchemaSyntax.getSimpleType(c); // may throw IllegalArg
    }

    public <T> Stringer<T> getStringer(Class<T> c) {
	return classToStringerMap.get(c);
    }
    protected <T> void setStringer(Class<T> c, Stringer<T> s) {
	classToStringerMap.put(c, s);
    }

    protected void initStringConversions() {

	// Date
	setStringer(Date.class, new Stringer<Date>() {
	    ISODateFormat format = new ISODateFormat();
	    public String toString(Date d) {
		return format.formatDateTime(d);
	    }
	    public Date fromString(String text) {
		return format.parseDateTime(text);
	    }
	    public String xmlSchemaDatatype() {
		return "dateTime";
	    }
	});

	// Duration
	setStringer(Duration.class, new Stringer<Duration>() {
	    public String toString(Duration d) {
		return d.toISOString();
	    }
	    public Duration fromString(String text) {
		return new Duration(text);
	    }
	    public String xmlSchemaDatatype() {
		return "duration";
	    }
	});

	// URI
	setStringer(URI.class, new Stringer<URI>() {
	    public String toString(URI uri) {
		return uri.toString();
	    }
	    public URI fromString(String text) {
		try {
		    return new URI(text);
		}
		catch (URISyntaxException e) {
		    throw new RethrownException(e);
		}
	    }
	    public String xmlSchemaDatatype() {
		return "anyURI";
	    }
	});

	// URL
	setStringer(URL.class, new Stringer<URL>() {
	    public String toString(URL url) {
		return url.toString();
	    }
	    public URL fromString(String text) {
		try {
		    return new URL(text);
		}
		catch (MalformedURLException e) {
		    throw new RethrownException(e);
		}
	    }
	    public String xmlSchemaDatatype() {
		return "anyURI"; 			// ??? /\/
	    }
	});

	// Color
	setStringer(Color.class, new Stringer<Color>() {
	    public String toString(Color c) {
		return Integer.toString(c.getRGB());
	    }
	    public Color fromString(String text) {
		try {
		    return Color.decode(text);
		}
		catch (NumberFormatException e) {
		    throw new RethrownException(e);
		}
	    }
	    public String xmlSchemaDatatype() {
		return "int"; 				// /\/ ???
	    }
	});

    }

    /*
     * Finding relevant classes
     */

    /**
     * Returns a recursively composed list of classes that are related
     * to the specified class by being the types of fields, etc.
     */
    public List<Class> relevantClasses(Class c) {
	return relevantClasses(Lisp.list(c));
    }

    /**
     * Returns a recursively composed list of classes that are related
     * to the classes in the specified array by being the types of
     * fields, etc.
     */
    public List<Class> relevantClasses(Class[] topClasses) {
	return relevantClasses(Arrays.asList(topClasses));
    }

    /**
     * Returns a recursively composed list of classes that are related
     * to the classes in the specified list by being the types of
     * fields, etc.
     */
    public List<Class> relevantClasses(List<Class> topClasses) {
	List<Class> relevantClasses = new LinkedList<Class>();
	for (Class c: topClasses) {
	    collectRelevantClasses(c, relevantClasses, Lisp.NIL);
	}
	relevantClasses.remove(Object.class);	// /\/
	return relevantClasses;
    }

    /**
     * Returns a recursively composed list of classes that are related
     * to the classes in the specified list by being the types of
     * fields, etc.  This method retains the order of the original
     * list, but does preorder expansion of elements to include
     * any relevant classes not already there.
     */
    public List<Class> expandRelevantClasses(List<Class> topClasses) {
	// Debug.noteln("Expanding relevant classes in", topClasses);
	List<Class> collect = new LinkedList<Class>();
	for (LList tail = LList.newLList(topClasses);
	     !tail.isEmpty(); tail = tail.cdr()) {
	    Class c = (Class)tail.car();
	    Debug.expect(!collect.contains(c));
	    collectRelevantClasses(c, collect, tail.cdr());
	}
	collect.remove(Object.class); 		// /\/
	// Debug.noteln("Expanded to", collect);
	return collect;
    }

    protected void collectRelevantClasses(Class c, List result, List tail) {
	if (c == null || result.contains(c) || tail.contains(c))
	    return;
	result.add(c);

	ClassDescr cd = getClassDescr(c);
	if (cd.isStruct()) {
	    for (Iterator fi = cd.getFieldDescrs().iterator(); fi.hasNext();) {
		FieldDescr fd = (FieldDescr)fi.next();
		collectRelevantClasses(fd.getTypeDescr(), result, tail);
	    }
	}
    }

    protected void collectRelevantClasses
	           (ClassDescr ftype, List result, List tail) {
	if (ftype == null)
	    return;
	else if (ftype.isCollection())
	    collectRelevantClasses(ftype.getEltType(), result, tail);
	else if (ftype.isMap()) {
	    collectRelevantClasses(ftype.getKeyType(), result, tail);
	    collectRelevantClasses(ftype.getValType(), result, tail);
	}
	else
	    collectRelevantClasses(ftype.getDescribedClass(), result, tail);
    }


    /*
     * Making FieldDescrs
     */

    /**
     * Called from a ClassDescr to make descriptions of the class's
     * fields.  This method should not normally be used for other
     * purposes.
     */
    FieldMap makeFieldDescrs(Class c) { // deliberately not public
	return collectFieldInfo(c);
    }

    /*
     * Determining the visible fields of a class
     */

    protected FieldMap collectFieldInfo(Class objClass) {
	// Get a preliminaty list based on the field declarations
	// Entires will be removed if suitable "get" and "set"
	// methods cannot be found.
	FieldMap fields = collectFields(objClass, new FieldMap());

	// An annotation allows some fields to be hidden.
	HiddenFields hidden =
	    //\/ don't know why we need the cast here
	    (HiddenFields)objClass.getAnnotation(HiddenFields.class);

	if (hidden != null) {
	    String[] names = hidden.value();
	    for (int i = 0; i < names.length; i++) {
		fields.remove(fields.fieldForName(names[i]));
	    }
	}

	// Another annotation allows a field to appear under a
	// different name and use "get" and "set" methods for that
	// name.
	AliasFields alias =
	    (AliasFields)objClass.getAnnotation(AliasFields.class);

	if (alias != null) {
	    String[] renames = alias.value();
	    for (int i = 0; i < renames.length; i++) {
		String[] fromTo = Strings.breakAtFirst("->", renames[i]);
		String from = fromTo[0];
		String to = fromTo[1];
		fields.rename(from, to, this);
	    }
	}

	// Look at the methods, starting with objClass and then
	// trying successive superclasses, taking the first
	// suitable "get" and "set" methods for each field.
	for (Class atClass = objClass; atClass != null;) {

	    // Look at the declared methods of the class
	    // Method[] methods = atClass.getDeclaredMethods();
	    Method[] methods = getDeclaredMethods(atClass);
	    for (Method meth: methods) {
		String methName = meth.getName();
		if (isGetName(methName))
		    tryGetMethod(meth, fields);
		else if (isSetName(methName)) {
		    trySetMethod(meth, fields);
		}
	    }

	    // Look at superclass methods
	    atClass = atClass.getSuperclass();
	}

	// Remove fields that don't have both "get" and "set",
	// and set the typeDescr of ones that do.
	fields.removeIf(new Predicate1<FieldDescr>() {
	    public boolean trueOf(FieldDescr fd) {
		return fd.getter == null || fd.setter == null;
	    }
	});
	for (FieldDescr fd: fields.getFields()) {
	    fd.typeDescr = makeFieldTypeDescr(fd, objClass);
	}

	return fields;

    }

    /**
     * Constructs a {@link ClassDescr} that describes the value of
     * a field.
     */
    protected ClassDescr makeFieldTypeDescr(FieldDescr fd, Class fromClass) {
	Class fieldClass = fd.getType();

	// If fieldClass is a List or Map, we'd like to know
	// the content types (elt, or key + val).

	// For a TypedList class, makeClassDescr(Class) will ask
	// for its element class.

	// See if we can determine the element class of a collection.
	if (Collection.class.isAssignableFrom(fieldClass)) {

	    // There's an annotation that also tells us the element class.
	    ElementClass ec = fd.getAnnotation(ElementClass.class);
	    if (ec != null)
		return makeClassDescr(fieldClass, ec.value());

	    // Generics can also provide an element class.
	    Class eltClass = fd.determineElementType();
	    if (eltClass != null)
		return makeClassDescr(fieldClass, eltClass);

	}

	return getClassDescr(fieldClass); // also handles TypedLists

    }

    /**
     * Returns the class's declared methods, or an empty array if
     * there's a security exception (as there might be in an applet).
     */
    protected Method[] getDeclaredMethods(Class c) {
        // It's the Object class that usually causes problems in applets.
        if (c == Object.class)
            return new Method[]{};

	try {
	    return c.getDeclaredMethods();
	}
	catch (SecurityException e) {
	    Debug.forceln("Problem getting methods of " + c);
	    Debug.noteException(e, false);
	    return new Method[]{};
	}

    }

    protected void tryGetMethod(Method meth, FieldMap fields) {
	Class[] ptypes = meth.getParameterTypes();
	int mod = meth.getModifiers();
	// Must be a public, 0-argument method
	if (!(Modifier.isPublic(mod) && ptypes.length == 0))
	    return;
	// Find corresponding field, if any
	String fieldName = fieldNameFromGetName(meth.getName());
	FieldDescr fd = fields.fieldForName(fieldName);
	// The field must exist and not already have a "get" method ...
	if (fd != null && fd.getter == null) {
	    // ... and this method must return the right type
	    Class fieldType = fd.getType();
	    if (meth.getReturnType() == fieldType)
		fd.getter = meth;
	}
    }

    protected void trySetMethod(Method meth, FieldMap fields) {
	Class[] ptypes = meth.getParameterTypes();
	int mod = meth.getModifiers();
	// Must be a public, 1-argument method
	if (!(Modifier.isPublic(mod) && ptypes.length == 1))
	    return;
	// Find corresponding field, if any
	String fieldName = fieldNameFromSetName(meth.getName());
	FieldDescr fd = fields.fieldForName(fieldName);
	// The field must exist and not already have a "set" method ...
	if (fd != null && fd.setter == null) {
	    // ... and this method must have an argument of the right type
	    Class fieldType = fd.getType();
	    // /\/: Must return-type be void?
	    if (ptypes[0] == fieldType)
		fd.setter = meth;
	}
    }

    public static final boolean isGetName(String name) {
	return name.length() > 3 && name.startsWith("get");
    }

    public static final boolean isSetName(String name) {
	return name.length() > 3 && name.startsWith("set");
    }

    public static final String fieldNameFromGetName(String name) {
	return Strings.uncapitalize(name.substring(3));
    }

    public static final String fieldNameFromSetName(String name) {
	return Strings.uncapitalize(name.substring(3));
    }

    /** Returns an initial List of {@link FieldDescr}s. */
    protected FieldMap collectFields(Class c, FieldMap fields) {

	// We look at fields in a class before looking at
	// its superclass, and keep track of names we've
	// already seem, so that a class's fields will hide
	// any inherited fields of the same name.  That ensures
	// the the most "local" field's type will be used.

	// We don't process interfaces, because they can provide
	// only static finals.

	// Non-inherited fields from this class.
	// Field[] localf = c.getDeclaredFields();
	Field[] localf = getDeclaredFields(c);
	for (Field f: localf)
	    if (isWanted(f))
		if (fields.fieldForName(f.getName()) == null)
		    fields.add(makeFieldDescr(f));

	// Recursively process superclass, if it exists
	Class sup = c.getSuperclass();
	if (sup != null)
	    collectFields(sup, fields);

	return fields;
    }

    protected FieldDescr makeFieldDescr(Field f) {
	return new FieldDescr(this, f);
    }

    /**
     * Returns the class's declared fields, or an empty array if
     * there's a security exception (as there might be in an applet).
     */
    protected Field[] getDeclaredFields(Class c) {
        // It's the Object class that usually causes problems in applets.
        if (c == Object.class)
            return new Field[]{};

	try {
	    return c.getDeclaredFields();
	}
	catch (SecurityException e) {
	    Debug.forceln("Problem getting fields of " + c);
	    Debug.noteException(e, false);
	    return new Field[]{};
	}

    }

    protected boolean isWanted(Field f) {
	int m = f.getModifiers();
	return !(Modifier.isFinal(m) || Modifier.isStatic(m) ||
		 Modifier.isTransient(m));
    }


    /*
     * Testing
     */

    /**
     * Prints a description of a class on <code>System.out</code>.
     */
    public void describeClass(Class c) {
	describeClass(c, System.out);
    }

    /**
     * Prints a description of a class on the designated PrintStream.
     */
    public void describeClass(Class c, PrintStream out) {
	ClassDescr cd = getClassDescr(c);
	List fields = cd.getFieldDescrs();

	out.println(cd.toString());
	out.println("Java name: " + cd.theClass.getName());
	out.println("External name: " + cd.getExternalName());
	out.println(fields.isEmpty() ? "No fields" : "Fields:");
	for (Iterator i = fields.iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    out.println(fd.getName() + ": " + fd.getTypeDescr());
	    out.println("   getter: " + fd.getter);
	    out.println("   setter: " + fd.setter);
	}
    }

    /**
     * Test loop that repeatedly asks the user for a class name
     * and prints a description.
     */
    public static void main(String[] argv) {
	ClassSyntax syntax = new ClassSyntax();
	for (;;) {
	    String in = Util.askLine("Class name:");
	    if (in.equals("bye"))
		return;

	    Class c = syntax.classFinder.classForName(in);
	    System.out.println(c);
	    if (c == null) continue;

	    syntax.describeClass(c, System.out);
	    System.out.println("");
	}
    }

}
