/* File: MatchEnv.java
 * Contains: A class for pattern-matching
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: January 1998
 * Updated: Tue Jun 27 16:24:58 2006 by Jeff Dalton
 * Copyright: (c) 1998, 2005, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;

import ix.icore.Variable;	// used in assertions

import ix.util.*;
import ix.util.lisp.*;

/** 
 * A root class for objects that contain the bindings that result
 * from pattern-matching. <p>
 *
 * MatchEnvs are not used automatically in MatchTable-based matching;
 * they are merely available in case they are useful.  For instance, a
 * MatchCase tryMatch method might construct a MatchEnv while matching
 * and return it if the match succeeds.  It would then be passed to
 * the MatchCase's ifSelected method by the MatchTable.
 *
 * @see MatchTable
 * @see MatchCase
 */

public class MatchEnv extends HashMap {

    public MatchEnv() { super(); }

    public MatchEnv(Map t) { super(t); }

    public Object checkedPut(Object key, Object value) {
	Debug.expect(!(value instanceof ItemVar),
		     "Attempt to bind " + key + " to " + value);
	Debug.expect(key instanceof ItemVar || key instanceof Variable,
		     "Attempt to bind " + key + " to " + value);
	if (key instanceof Variable)
	    Debug.expect(((Variable)key).getValue() == null,
			 "Attempt to bind bound " + key + " to " + value);
	return super.put(key, value);
    }

    /**
     * Returns a Map containing any variable-to-value pairs specified
     * by this MatchEnv for instances of the specified variable class.
     *
     * @return a nonEmpty Map if any variable bindings were specified,
     *         otherwise null.
     */
    public Map getVariableBindings(Class variableClass) {
	Map bindings = null; 	// create if needed
	for (Iterator i = this.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object var = e.getKey();
	    if (variableClass.isInstance(var)) {
		if (bindings == null)
		    bindings = new HashMap();
		bindings.put(var, e.getValue());
	    }
	}
	return bindings;
    }

    public Object instantiate(Object obj, final Function1 ifUnbound) {
	ObjectCopier copier = new DeepCopier() {
	    public Object copy(Object obj) {
		return obj instanceof ItemVar
		    ? mapItemVar((ItemVar)obj)
	            : super.copy(obj);
	    }
	    Object mapItemVar(ItemVar var) {
		Object value = get(var);
		if (value == null)
		    return ifUnbound.funcall(var);
		else
		    return value;
	    }
        };
        return copier.copy(obj);
    }

    public Object instantiateTree(Object tree,
				  Function1 ifUnbound) {
	// Debug.noteln("instantiateTree(" + tree + ", " + ifUnbound + ")");
	if (tree instanceof ItemVar) {
	    ItemVar var = (ItemVar)tree;
	    Object val = get(var);
	    if (val == null)
		return ifUnbound.funcall(var);
	    else 
		return val;
	}
	else if (tree instanceof Cons) {
	    Cons c = (Cons)tree;
	    return new Cons(instantiateTree(c.car(), ifUnbound),
			    (LList)instantiateTree(c.cdr(), ifUnbound));
	}
	else
	    return tree;
    }

}

// Issues:
// * Does it make sense to have instantiateTree() here?

