/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Oct  2 22:15:44 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;

import ix.icore.Variable;
import ix.util.*;
import ix.util.lisp.Lisp;	// for debugging output
import ix.util.context.*;

/**
 * Manages a pool of constraints on variable values.  ... /\/
 *
 * <p>N.B. This is a simple-minded "proof of concept" version, lacking
 * many obvious optimisations.
 *
 * @see MatchChoice
 */
public class MatchChoiceManager {

    public static boolean traceDetails = false;

    protected LLQueue matchChoices = new LLQueue();

    protected ContextBoolean changedSinceRecalc = new ContextBoolean(false);

    public MatchChoiceManager() {
    }

    public void reset() {
	matchChoices.clearCompletely();
	changedSinceRecalc = new ContextBoolean(false);
    }

    public void clear() {
	matchChoices.clear();
	changedSinceRecalc.set(false);
    }

    public void add(MatchChoice mc) {
	Debug.expect(!matchChoices.contains(mc));
	matchChoices.add(mc);
	changedSinceRecalc.set(true);
    }

    public void remove(MatchChoice mc) {
	if (!matchChoices.remove(mc))
	    throw new ConsistencyException("Can't remove if not present.");
	changedSinceRecalc.set(true);
    }

    public Set getVariables() {
	Set result = new TreeSet();
	for (Iterator i = matchChoices.iterator(); i.hasNext();) {
	    MatchChoice mc = (MatchChoice)i.next();
	    result.addAll(mc.getVariables());
	}
	return result;
    }

    /**
     * Returns a map: variable -> set of possible values, one entry
     * for each variable known to this manager.
     *
     * @see #getVariables()
     * @see #getPossibleValues(Set)
     */
    public Map getPossibleValues() {
	return getPossibleValues(getVariables());
    }

    /**
     * Returns a map: variable -> set of possible values, one entry
     * for each variable in the specified set.
     */
    public Map getPossibleValues(Set vars) {
	Debug.expect(!changedSinceRecalc.isTrue());
	Map result = new TreeMap();
	for (Iterator i = vars.iterator(); i.hasNext();) {
	    Variable v = (Variable)i.next();
	    Set values = getPossibleValues(v);
	    result.put(v, values);
	}
	return result;
    }

    public Set getPossibleValues(Variable v) {
	Debug.expect(!changedSinceRecalc.isTrue());
	Set result = null;
	for (Iterator i = matchChoices.iterator(); i.hasNext();) {
	    MatchChoice mc = (MatchChoice)i.next();
	    if (mc.getVariables().contains(v)) {
		Set possible = mc.getPossibleValues(v);
		if (result == null)
		    result = possible;
		else
		    Debug.expect(possible.equals(result),
				 "Values " + possible + " !equals " + result);
	    }
	}
	return result;
    }

    public void recalculate() {
	for (ListIterator i = matchChoices.listIterator(); i.hasNext();) {
	    MatchChoice c_i = (MatchChoice)i.next();
	    c_i.resetForChecks();
	}
	List active = new ArrayList(matchChoices);
	List changed = new LinkedList();
	while (!active.isEmpty()) {
	    for (Iterator i = active.iterator(); i.hasNext();) {
		MatchChoice c_i = (MatchChoice)i.next();
		//\/ Need to visit only those that c_i constrains,
		// but we don't keep a record of which those are.
		for (Iterator j = matchChoices.iterator(); j.hasNext();) {
		    MatchChoice c_j = (MatchChoice)j.next();
		    if (c_i != c_j
			  && Collect.haveCommonElements
			        (c_i.getVariables(), c_j.getVariables())) {
			if (c_j.shrinkToFit(c_i))
			    changed.add(c_j);
		    }
		}
	    }
	    active = changed;
	    changed = new LinkedList();
	}
	changedSinceRecalc.set(false);
	noticeForcedBindings();
    }

    // /\/: What about variables that have NO possible value?

    protected MatchEnv noticeForcedBindings() {
	// /\/: Returns a value only for use in subclass VariableManager.
	// /\/: We ought to save the variables set and possible values map.
	// Maybe have a class that contains recalculated "state" as fields.
	MatchEnv forced = new MatchEnv();
	Map possibilities = getPossibleValues();
	for (Iterator i = possibilities.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Variable var = (Variable)e.getKey();
	    Set values = (Set)e.getValue();
	    if (!var.isBound() && values.size() == 1)
		// /\/: There's no get(index) method for Sets.
		forced.put(var, values.iterator().next());
	}
	if (!forced.isEmpty())
	    Debug.noteln("Forced bindings", forced);
	return forced;
    }

    public void showState() {
	Debug.noteln("MatchChoiceManager state in", Context.getContext());
	for (Iterator i = matchChoices.iterator(); i.hasNext();) {
	    MatchChoice mc = (MatchChoice)i.next();
	    Debug.noteln(Lisp.hashName(mc));
	    Debug.noteln("  variables", mc.getVariables());
	    Debug.noteln("  branches", mc.getBranches());
	    Debug.noteln("  live    ", mc.getLiveBranches());
	    Debug.noteln("");
	}

    }

}
