/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Nov 11 00:30:29 2004 by Jeff Dalton
 * Copyright: (c) 2003, 2004, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;

import ix.icore.Variable;
import ix.icore.domain.Constraint;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.*;

public class MatchChoice {

    public static final Symbol
	S_VARIABLE = Symbol.intern("variable"),
	S_MATCH_CHOICE = Symbol.intern("match-choice");

    protected List branches = new ArrayList(); // List of Bindings
    protected Set variables = new TreeSet();

    protected ContextValue __liveBranches =
	new TypedContextValue(List.class, null);

    public MatchChoice() {
    }

    public MatchChoice(List initialBranches) {
	for (Iterator i = initialBranches.iterator(); i.hasNext();) {
	    Debug.expect(i.next() instanceof Bindings);
	}
	branches.addAll(initialBranches);
	computeVariableSet();
    }

    public MatchChoice(Constraint c) {
	if (c.getType() != S_VARIABLE || c.getRelation() != S_MATCH_CHOICE)
	    throw new IllegalArgumentException
		("Not a varibale match-choice constraint");
	if (c.getParameter(0) instanceof Map)
	    throw new IllegalArgumentException("Obsolete constraint format");
	final List vars = (List)c.getParameter(0);
	List images = (List)c.getParameter(1);
	List maps = (List)Collect.map(images, new Function1() {
	    public Object funcall(Object image) {
		// Return a map that pairs each elt of vars with the
		// corresponding elt of the image
		return Collect.extendMap(new MatchEnv(), vars, (List)image);
	    }
	});
	branches.addAll(Bindings.mapsToBindings(maps));
	computeVariableSet();
    }

    public Constraint toConstraint() {
	// /\/: Do we care about the order? unboundVarsIn returns
	// a HashSet rather than a TreeSet.
	Set unbound = new TreeSet(Variable.unboundVarsIn(variables));
	final List vars = new LinkedList(unbound);
	List maps = Bindings.bindingsToMaps(branches);
	List images = (List)Collect.map(maps, new Function1() {
	    public Object funcall(Object map) {
		// Return a list of the variable values given by the map
		return Collect.map(vars, Collect.toFunction((Map)map));
	    }
	});
	return new Constraint(S_VARIABLE,
			      S_MATCH_CHOICE, 
			      Lisp.list(vars, images));
    }

    List getBranches() {
	return branches;
    }

    public Set getVariables() {
	return variables;
    }

    Set getPossibleValues(Variable var) {
	Set values = new SortedValueSet();
	for (Iterator i = getLiveBranches().iterator(); i.hasNext();) {
	    Bindings b = (Bindings)i.next();
	    Object val = b.valueOf(var);
	    if (val != null)
		values.add(val);
	}
	return values;
    }

    public List getLiveBranches() {
	// /\/: Should be called externally only after the choice manager
	// has recalculated.  Internal calls during recalc are ok.
	return (List)__liveBranches.get();
    }

    protected void computeVariableSet() {
	variables.clear();
	for (Iterator i = branches.iterator(); i.hasNext();) {
	    Bindings b = (Bindings)i.next();
	    variables.addAll(b.getVariables());
	}
    }

    void resetForChecks() {
	__liveBranches.set(new LinkedList(branches));
    }

    boolean shrinkToFit(MatchChoice c) {
	if (MatchChoiceManager.traceDetails)
	    Debug.noteln(Lisp.hashName(this) + " shrinking to fit " +
			 Lisp.hashName(c));
	boolean changed = false;
	for (Iterator i = getLiveBranches().iterator(); i.hasNext();) {
	    Bindings b = (Bindings)i.next();
	    if (!b.isConsistentWith(c)) {
		i.remove();
		changed = true;
	    }
	}
	return changed;
    }

}
