/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 12 15:36:59 2008 by Jeff Dalton
 */

package ix.util.lisp;

import java.io.Serializable;
import java.util.*;
import ix.util.*;


/**
 * The root class for Lisp symbols.
 *
 * <p>The correct way to create a new symbol is to call 
 *    {@link #intern(String name)}.
 * It will return an instance of Symbol or of an appropriate subclass,
 * typically depending on the first character of the name.</p>
 */

// At least that's how things are for now.  It's not yet clear what's best.

public class Symbol implements LispObject, SemiPrimitive,
			       Comparable, Serializable {

    /**
     * The mapping from names to Symbols used to ensure that
     * there is at most one symbol with any given name.
     */
    protected static Hashtable obTable = new Hashtable();

    /**
     * This Symbol's name.
     */
    protected String name;

    /**
     * Constructs a Symbol with the specified name and records it
     * as the Symbol that corresponds to that name.  This Symbol
     * will then be returned by {@link #intern(String)} whenever it
     * is given that name.  This constructor should not normally
     * be called directly.  Call {@link #intern(String)} instead.
     */
    protected Symbol(String name) {
	this.name = name;
	Debug.expect(obTable.get(name) == null, "symbol already exists", name);
	obTable.put(name, this);
    }

    /**
     * Returns the Symbol that has the specified string as its name,
     * creating a new one only if one does not already exist.  Thus,
     * there is always at most one Symbol with a given name.
     * This is a factory method that should be called instead of
     * calling any constructor directly.  It returns an instance
     * of Symbol, or of an appropriate subclass of Symbol,
     * depending on the characters in the name.
     * <ul>
     * <li>{@link Keyword} if the name begins with ":".
     * <li>{@link ItemVar} if the name begins with "?".
     * <li>{@link NumberlikeSymbol} if the name would normally
     *     be parsed as an integer or floating-point number.
     * <li>{@link DelimitedSymbol} if the name would not normally
     *     be parsed as a Symbol because it contains whitespace,
     *     parentheses, or other syntactically significan characters.
     * <li>{@link Symbol} in all other cases.
     * </ul>
     */
    public static synchronized Symbol intern(String s) {
	Symbol sym = (Symbol)obTable.get(s);
	if (sym != null)
	    return sym;
	else if (s.length() == 0)
	    return new DelimitedSymbol(s);
	else {
	    char ch;
	    switch (ch = s.charAt(0)) {
	      case ':' : return new Keyword(s);
  	      case '?' : return new ItemVar(s);
	      case '+' : return maybeNumber(s);
	      case '-' : return maybeNumber(s);
	      case '.' : return maybeNumber(s);
	      default  : return Character.isDigit(ch) ? maybeNumber(s)
			        : maybeSpecial(s);
	    }
	}
    }

    /**
     * A simple, faster version of {@link #intern(String)} that
     * is used internally when it's known that the string does
     * not contain any characters that need escaping.
     */
    static synchronized Symbol quickIntern(String s) {
	Symbol sym = (Symbol)obTable.get(s);
	if (sym != null)
	    return sym;
	else if (s.length() == 0)
	    return new DelimitedSymbol(s);
	else {
	    char ch;
	    switch (ch = s.charAt(0)) {
	      case ':' : return new Keyword(s);
  	      case '?' : return new ItemVar(s);
	      default  : return new Symbol(s);
	    }
	}
    }

    private static Symbol maybeNumber(String s) {
	try { Long.valueOf(s); return new NumberlikeSymbol(s); }
	catch (NumberFormatException e1) {
	    try { Double.valueOf(s); return new NumberlikeSymbol(s); }
	    catch (NumberFormatException e2) {
		// If we get here, the string didn't parse as
		// a long or a double.
		return maybeSpecial(s);
	    }
	}
    }

    private static Symbol maybeSpecial(String s) {
	int len = s.length();
	for (int i = 0; i < len; i++) {
	    char c = s.charAt(i);
	    // N.B. The test below must agree with the LispStreamTokenizer.
	    if (c == ' ' || c == '\n' || c == '\r' ||
		c == '"' || c == '|'  || c == '\\' ||
		c == '(' || c == ')'  || c == '['  || c == ']' ||
		c == ',' || c == ';')
	      return new DelimitedSymbol(s);
	}
	return new Symbol(s);
    }

    /**
     * Returns true iff a symbol with the specified name exists.
     */
    public static synchronized boolean exists(String name) {
	return obTable.get(name) != null;
    }

    public int compareTo(Object o) {
	if (o instanceof Symbol)
	    return name.compareTo(((Symbol)o).name);
	else
	    throw new ClassCastException
		("Cannot compare symbol " + this + " to " + o);
    }

    protected Object readResolve() throws java.io.ObjectStreamException {
	// Deserialization magic to avoid multiple (visible)
	// instances of Symbols that should be eq.  The instances
	// are created (it seems) but then this method is called ...
	return intern(this.name);
    }

    public String toString() {
	return name;
    }

    public static Symbol valueOf(String name) {
	// For the SemiPrimitive interface
	return intern(name);
    }

}
