/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Dec  7 16:57:08 2007 by Jeff Dalton
 * Copyright: (c) 2005, 2007, AIAI, University of Edinburgh
 */

package ix.util.lisp;

import java.util.*;

import ix.iscript.IScript;
import ix.iscript.Expression;
import ix.iscript.ListOfExpression;
import ix.iscript.LinkedListOfExpression;

import ix.util.*;
import ix.util.xml.XML;		// for debugging output

/**
 * The skeleton of a progamming language interpreter.  This class
 * implements a language that is semantically similar to a simple
 * version of Scheme without continuations.  However, it doesn't
 * provide any syntax or built-in functions.  It is expected that
 * they will be provided by a subclass.  The subclass should produce
 * {@link Interpreter.Expr Expr} objects from whatever surface syntax
 * it desires.  There are Expr classes for constants (literals),
 * variable references, conditional expressions, and so on.  Functions
 * should implement the {@link Interpreter.Function Function} interface.
 * Usually they will extend the {@link Interpreter.JFunction JFunction} class.
 */
public abstract class Interpreter {

    /**
     * The value this interpreter thinks expressions should return
     * when they cannot determine any more appropriate value.
     */
    protected Object DEFAULT = null;

    /**
     * The value that an interpreter thinks should be used
     * by default to represent truth.  It is the same as Boolean.TRUE.
     */
    protected static Object TRUE = Boolean.TRUE;

    /**
     * The value that an interpreter thinks should be used
     * by default to represent falsity.  It is the same as Boolean.FALSE.
     */
    protected static Object FALSE = Boolean.FALSE;

    /**
     * This interpreter's global / top-level environment.
     */
    protected GlobalEnv globalEnv = new GlobalEnv();

    /**
     * An IScript factory used when constructing external forms.
     */
    protected static IScript iscript = new IScript();

    /**
     * Controls whether only safe builtin functions are added.
     * To make a safe interpreter, this field must be set to true
     * before any {@link Interpreter.JFunction JFunction} is
     * added.
     *
     * @see #isSafe()
     */
    protected boolean acceptOnlySafeBuiltins = false;

    /**
     * Constructs an interpreter.
     */
    public Interpreter() {
    }

    /**
     * Determines whether this interpreter regards the object
     * as true; otherwise, the object is considered false.
     *
     * <p>Note that expressions do not call this method directly.
     * Instead, they must call the {@link Env#isTrue(Object)} method
     * of an environment.</p>
     */
    public boolean isTrue(Object obj) {
	return !obj.equals(Boolean.FALSE);
    }

    /**
     * Evaluates the expression in the top-level enviromnent
     * of this interpreter.
     */
    public Object topLevelEval(Expr e) {
	return e.evalIn(globalEnv);
    }

    /**
     * Applies the function to the specified arguments.
     * Note that the function may be given as a name ({@link Symbol})
     * rather than as an instance of {@link Function}.
     * If so, the name is looked up in the top-level environment
     * of this interpreter.
     */
    public Object topLevelApply(Object function, Object[] args) {
	if (function instanceof Function) {
	    Function fn = (Function)function;
	    fn.checkArity(args.length);
	    return fn.applyTo(args);
	}
	else if (function instanceof Symbol) {
	    Object fval = globalEnv.lookup((Symbol)function);
	    if (fval instanceof Function) {
		Function fn = (Function)fval;
		fn.checkArity(args.length);
		return fn.applyTo(args);
	    }
	}
	throw new IllegalArgumentException
	    (function + " is not defined as a function.");
    }

    /**
     * Says whether the interpreter can safely be used in cases where
     * you don't want the user to be able to do such things as call
     * arbitrary Java methods, read or write files, or get other
     * information about the host machine.  Interpreters are unsafe
     * by default.
     *
     * @see #acceptOnlySafeBuiltins
     * @see LispInterpreter#LispInterpreter(boolean beSafe)
     */
    public boolean isSafe() {
	return acceptOnlySafeBuiltins;
    }


    /**
     * A utility that returns a relatively readable name for a class.
     */
    protected static String nameForClass(Class c) {
	return Strings.afterLast(".", c.getName());
    }

    /**
     * The context in which an expression is evaluated.
     */
    public static interface Env {

	/**
	 * Returns the value bound to the variable by this environment.
	 *
	 * @throws Error if the variable is not bound.
	 */
	public Object lookup(VarRef ref);

	/**
	 * Returns a new Env in which the variables are
	 * bound to the corresponding values.
	 *
	 * @throws Error if there aren't exactly as many variables as values.
	 */
	public Env bind(Symbol[] vars, Object[] vals);

	/**
	 * Changes the value bound to the variable and then
	 * returns the new value.  Note that this typically modifies
	 * an existing binding relationship rather than create a
	 * new one; however, a new one may be created, for instance
	 * if the variable is not already bound.
	 */
	public Object assign(VarRef ref, Object val);

	/**
	 * Returns the value that should be returned by expressions
	 * that do not have anything more appropriate to return.
	 */
	public Object defaultValue();

	/**
	 * Determines whether the object should be regarded as true;
	 * otherwise, the object is considered false.
	 */
	public boolean isTrue(Object value);

    }

    /**
     * An implementation of Env that supplies some utility methods.
     * It is not a static class and is therefore always associated
     * with an Interpreter.  This makes it a way to access some
     * properies of the interpreter, in particular its default
     * value and {@link Interpreter#isTrue(Object) isTrue} method.
     */
    protected abstract class AbstractEnv implements Env {
	public AbstractEnv() { }
	/** Returns its Interpreter's default value. */
	public Object defaultValue() {
	    return DEFAULT;
	}
	/** Calls its Interpreter's isTrue method. */
	public boolean isTrue(Object value) {
	    return Interpreter.this.isTrue(value);
	}
    }

    /**
     * An object that represents a function that can be applied
     * to arguments.
     */
    public static interface Function {

	public static final int ANY_ARITY = -1; // means any number of args

	/**
	 * Returns the result of applying this function to the
	 * specified arguments.  This is the method that should
	 * be defined in order to implement whatever the function
	 * is meant to do.
	 */
	public Object applyTo(Object[] args);

	/**
	 * Returns the number of arguments this function expects.
	 * If the function can take any number of arguments,
	 * the value is {@link Function#ANY_ARITY}.
	 */
	public int getArity();

	/**
	 * Checks whether this function can accept the indicated
	 * number of arguments.
	 *
	 * @throws Error if the function cannot accept that many arguments.
	 */
	public void checkArity(int numberOfArgs);

    }

    /**
     * An implementation of Function that supplies some utility methods
     * and implements part of the interface.
     */
    public static abstract class AbstractFunction implements Function {

	public AbstractFunction() { }

	public void checkArity(int numberOfArgs) {
	    int arity = getArity();
	    if (arity == Function.ANY_ARITY || arity == numberOfArgs)
		return;
	    else
		throw new Error("Expected " + arity + " arguments," +
				" but found " + numberOfArgs +
				" when calling " + this);
	}

	/**
	 * Returns an object if it is an instance of the specified class,
	 * but throws a ClassCastException if it isn't.  This allows us
	 * to provide a more infomative error message than if we'd just
	 * cast the object to the desired class.
	 */
	protected <T> T mustBe(Class<T> c, Object obj) {
	    return Util.mustBe(c, obj);
	}

    }

    /**
     * A function implemented in Java.
     *
     * <p>This class makes <code>isSafe()</code> return true;
     * a subclass should override that method if it is unsafe.
     */
    public static abstract class JFunction extends AbstractFunction {

	protected Symbol name;
	protected int arity;

	public JFunction(String name, int arity) {
	    this(Symbol.intern(name), arity);
	}

	public JFunction(Symbol name, int arity) {
	    this.name = name;
	    this.arity = arity;
	}

	public int getArity() {
	    return arity;
	}

	/**
	 * Says whether the function can safely be used in cases where
	 * you don't want the user to be able to read or write files
	 * or to get other information about the host machine.
	 */
	public boolean isSafe() {
	    return true;
	}

	public String toString() {
	    return "Function[" + name + "]";
	}

    }

    /**
     * The root class for expressions.  Expressions are able to
     * evaluate themselves when given an environment.  This is done
     * by calling the expression's {@link #_evalIn(Interpreter.Env)} method.
     * However, {@link #evalIn(Interpreter.Env)} should normally be called
     * instead.  It puts some debugging support around a call to _evalIn,
     * primarily so that a backtrace will appear if an exception
     * is thrown.

     * <p>Expressions have a "description" object that is
     * used by their {@link #toString()} method.  It's main
     * purpose is to describe the expression in a backtrace.</p>
     *
     * <p>Expressions also have an "external form".  Unlike the
     * description, it is not meant to be nicely readable by a
     * user.  Instead, it is a representation that can be
     * understood by the reflection and XML utilities and
     * can be input and output.  Converting an external form
     * back to an expression should produce an expression
     * that is equivalent to the original.</p>
     *
     * <p>The default external form for instances of subclasses of
     * this class is an IScript expression, an instance of a
     * subclass of {@link ix.iscript.Expression}.  There is
     * a fairly direct correspondence between subclasses of
     * that class and subclasses of this one.</p>
     *
     * <p>Conversion to external form is done by calling
     * the expression's {@link #externalForm()} method.
     * Coversion in the other direction is done by creating
     * an {@link ix.iscript.IScriptParser IScriptParser}
     * and calling its parseExpression method.</p>
     *
     * <p>This class also provides some utility methods.<p>
     */
    public static abstract class Expr {

	protected Object description;

	public Expr() {}

	/**
	 * Implements this expression's semantics.
	 *
	 * <p>Note that this method is usually not called directly.
	 * Instead, call {@link #evalIn(Interpreter.Env)}.</p>
	 */
	protected abstract Object _evalIn(Env env);

	/**
	 * Returns a representation of this expression that
	 * can be understood by the reflection and XML utilities.
	 *
	 * @see IScript
	 */
	public abstract Expression externalForm();

	public void setDescription(Object d) {
	    this.description = d;
	}

	public String toString() {
	    return description != null ? description.toString()
		:  super.toString();
	}

	/**
	 * Provides public access to this expression's semantics.
	 * This method typically wraps some debugging support around
	 * a call to {@link #_evalIn(Interpreter.Env)}.
	 */
	public Object evalIn(Env env) {
//  	    Debug.noteln("Evaluating:", XML.objectToXMLString(externalForm()));
	    Object result = null;
	    try {
		result = _evalIn(env);
		if (result == null)
		    throw new ConsistencyException
			("Null result from " + this);
		return result;
	    }
	    catch (ThrownResult r) {
		result = r.getResult();
		throw r;
	    }
	    finally {
		if (result == null) {
		    try {
			System.out.println("Backtrace: " + this);
		    }
		    catch (Throwable t) {
			Debug.displayException
			    ("Exception in eval 'finally' clause", t);
		    }
		}
	    }
	}

	/**
	 * A utility for evaluating argument expressions.
	 */
	protected Object[] evalArgs(Expr[] args, Env env) {
	    Object[] values = new Object[args.length];
	    for (int i = 0; i < args.length; i++) {
		values[i] = args[i].evalIn(env);
	    }
	    return values;
	}

	/**
	 * A utility for converting a List of {@link Expr}s
	 * to an array.
	 */
	protected static Expr[] makeExprArray(List exprList) {
	    Expr[] exprs = new Expr[exprList.size()];
	    int e = 0;
	    for (Iterator i = exprList.iterator(); i.hasNext();) {
		exprs[e++] = (Expr)i.next();
	    }
	    return exprs;
	}

	/**
	 * A utility for converting a List of variables ({@link Symbol}s)
	 * to an array.
	 */
	protected static Symbol[] makeVarArray(List varList) {
	    Symbol[] vars = new Symbol[varList.size()];
	    int v = 0;
	    for (Iterator i = varList.iterator(); i.hasNext();) {
		vars[v++] = (Symbol)i.next();
	    }
	    return vars;
	}

	/**
	 * A utility for converting an array of {@link Expr}s
	 * to a List of their external forms.
	 */
	protected static ListOfExpression externalForm(Expr[] exprs) {
	    ListOfExpression result = new LinkedListOfExpression();
	    for (int i = 0; i < exprs.length; i++)
		result.add(exprs[i].externalForm());
	    return result;
	}

    }

    /**
     * A reference to an object that is treated as a literal.
     * The value of this expression is simply that object.
     */
    public static class Literal extends Expr {
	Object value;
	public Literal(Object value) {
	    this.value = value;
	    this.description = value;
	}
	public Object evalIn(Env env) {
	    return value;	// direct, with no backtrace wrapping
	}
	protected Object _evalIn(Env env) {
	    return value;
	}
	public Expression externalForm() {
	    return iscript.makeLiteral(value);
	}
    }

    private static final int UNKNOWN = -2, GLOBAL = -1;

    /**
     * A reference to a variable.  The value of this expression
     * is the value bound to the variable by the environment this
     * expression is evaluated in.
     */
    public static class VarRef extends Expr {
	Symbol var;
	int up = UNKNOWN;
	int over = UNKNOWN;
	public VarRef(Symbol var) {
	    this.var = var;
	    this.description = var;
	}
	public Object evalIn(Env env) {
	    return env.lookup(this); // direct, with no backtrace wrapping
	}
	protected Object _evalIn(Env env) {
	    return env.lookup(this);
	}
	public Expression externalForm() {
	    return iscript.makeVarRef(var);
	}
    }

    /** A function call. */
    public static class Call extends Expr {
	Expr fnExpr;
	Expr[] argExprs;
	public Call(Expr fnExpr, List argExprList) {
	    this.fnExpr = fnExpr;
	    this.argExprs = makeExprArray(argExprList);
	}
	protected Object _evalIn(Env env) {
	    Object fval = fnExpr.evalIn(env);
	    if (fval instanceof Function) {
		Function fn = (Function)fval;
		Object[] args = evalArgs(argExprs, env);
		fn.checkArity(args.length);
		return fn.applyTo(args);
	    }
	    else
		throw new Error("Expected " + fnExpr + " to be a function, " +
				"but instead it was", fval);
	}
	public Expression externalForm() {
	    return iscript.makeCall(fnExpr.externalForm(),
				    externalForm(argExprs));
	}
    }

    /** An assignment to a variable. */
    public static class Assignment extends VarRef {
	Expr valueExpr;
	public Assignment(Symbol var, Expr valueExpr) {
	    super(var);
	    this.valueExpr = valueExpr;
	}
	public Object evalIn(Env env) {
	    // Eval directly, with no backtrace wrapping.
	    // N.B. We have to do this because VerRef redefines
	    // the method.
	    return env.assign(this, valueExpr.evalIn(env));
	}
	protected Object _evalIn(Env env) {
	    return env.assign(this, valueExpr.evalIn(env));
	}
	public Expression externalForm() {
	    return iscript.makeAssignment(var, valueExpr.externalForm());
	}
    }

    /** A simple conditional expression. */
    public static class If extends Expr {
	Expr test;
	Expr ifTrue;
	Expr ifFalse;
	public If(Expr test, Expr ifTrue, Expr ifFalse) {
	    this.test = test;
	    this.ifTrue = ifTrue;
	    this.ifFalse = ifFalse;
	}
	protected Object _evalIn(Env env) {
	    if (env.isTrue(test.evalIn(env)))
		return ifTrue.evalIn(env);
	    else
		return ifFalse.evalIn(env);
	}
	public Expression externalForm() {
	    return iscript.makeIf(test.externalForm(),
				  ifTrue.externalForm(),
				  ifFalse.externalForm());
	}
    }

    /**
     * A sequence of expressions.  The expressions are evaluated
     * in the order in which they were given when the sequence was
     * constructed, and the value of the last expression is returned.
     */
    public static class Sequence extends Expr {
	Expr[] exprs;
	public Sequence(List exprList) {
	    this.exprs = makeExprArray(exprList);
	}
	protected Object _evalIn(Env env) {
	    Object result = env.defaultValue();
	    for (int i = 0; i < exprs.length; i++) {
		result = exprs[i].evalIn(env);
	    }
	    return result;
	}
	public Expression externalForm() {
	    return iscript.makeSequence(externalForm(exprs));
	}
    }

    /**
     * A conditional AND.
     */
    public static class And extends Expr {
	Expr[] exprs;
	public And(List exprList) {
	    this.exprs = makeExprArray(exprList);
	}
	protected Object _evalIn(Env env) {
	    Object result = TRUE;
	    for (int i = 0; i < exprs.length; i++) {
		result = exprs[i].evalIn(env);
		if (!env.isTrue(result))
		    return result;
	    }
	    Debug.expect(env.isTrue(result));
	    return result;
	}
	public Expression externalForm() {
	    return iscript.makeAnd(externalForm(exprs));
	}
    }

    /**
     * A conditional OR.
     */
    public static class Or extends Expr {
	Expr[] exprs;
	public Or(List exprList) {
	    this.exprs = makeExprArray(exprList);
	}
	protected Object _evalIn(Env env) {
	    Object result = FALSE;
	    for (int i = 0; i < exprs.length; i++) {
		result = exprs[i].evalIn(env);
		if (env.isTrue(result))
		    return result;
	    }
	    Debug.expect(!env.isTrue(result));
	    return result;
	}
	public Expression externalForm() {
	    return iscript.makeOr(externalForm(exprs));
	}
    }

    /**
     * An expression that binds variables and then evaluates
     * an expression in the resulting environment.  That environment
     * is an extension of the one in which the Let is evaluated.
     */
    public static class Let extends Expr {
	Symbol[] vars;
	Expr[] valExprs;
	Expr body;
	public Let(List varList, List valExprList, Expr body) {
	    this.vars = makeVarArray(varList);
	    this.valExprs = makeExprArray(valExprList);
	    this.body = body;
	}
	protected Object _evalIn(Env env) {
	    return body.evalIn(env.bind(vars, evalArgs(valExprs, env)));
	}
	public Expression externalForm() {
	    return iscript.makeLet(vars, externalForm(valExprs),
				   body.externalForm());
	}
    }

    /**
     * An expression whose value is a function closed in the
     * environment in which this Lambda is evaluated.  It thus
     * implements lexical scoping.  The function is typically
     * a {@link Closure}.
     */
    public static class Lambda extends Expr {
	Symbol[] vars;
	Expr body;
	public Lambda(List varList, Expr body) {
	    this.vars = makeVarArray(varList);
	    this.body = body;
	}
	protected Object _evalIn(Env env) {
	    return new Closure(vars, body, env);
	}
	public Expression externalForm() {
	    return iscript.makeLambda(vars, body.externalForm());
	}
    }

    /**
     * An interpreted function.
     */
    public static class Closure extends AbstractFunction {
	Symbol[] vars;
	Expr body;
	Env definitionEnv;
	public Closure(Symbol[] vars, Expr body, Env definitionEnv) {
	    this.vars = vars;
	    this.body = body;
	    this.definitionEnv = definitionEnv;
	}
	public Object applyTo(Object[] args) {
	    try {
		return body.evalIn(definitionEnv.bind(vars, args));
	    }
	    catch (ThrownResult r) {
		return r.getResult();
	    }
	}
	public int getArity() {
	    return vars.length;
	}
	public String toString() {
	    return "Closure[" + Arrays.asList(vars) + ": " + body + "]";
	}
    }

    /** A while loop. */
    public static class While extends Expr {
	Expr test;
	Expr body;
	public While(Expr test, Expr body) {
	    this.test = test;
	    this.body = body;
	}
	protected Object _evalIn(Env env) {
	    Object result = env.defaultValue();
	    while (env.isTrue(test.evalIn(env))) {
		result = body.evalIn(env);
	    }
	    return result;
	}
	public Expression externalForm() {
	    return iscript.makeWhile(test.externalForm(), body.externalForm());
	}
    }

    /**
     * The root class for exceptions thrown for error conditions
     * detected by an Interpreter or by an expression.
     */
    public static class Error extends RuntimeException {
	public Error(String message) {
	    super(message);
	}
	public Error(String message, Object culprit) {
	    super(message + ": " + culprit);
	}
    }

    /**
     * A global / top-level environment.
     */
    public class GlobalEnv extends AbstractEnv {

	Map varToValMap = new HashMap();

	public GlobalEnv() { }

	public Object lookup(VarRef ref) {
	    Object val = varToValMap.get(ref.var);
	    if (val == null)
		throw new Error("Undefined variable", ref.var);
	    else
		return val;
	}

	public Object lookup(Symbol var) {
	    Object val = varToValMap.get(var);
	    if (val == null)
		throw new Error("Undefined variable", var);
	    else
		return val;
	}

	public Env bind(Symbol[] vars, Object[] vals) {
	    return new LexicalEnv(vars, vals, null);
	}

	public Object assign(VarRef ref, Object val) {
	    varToValMap.put(ref.var, val);
	    return val;
	}

	public Object assign(Symbol var, Object val) {
	    varToValMap.put(var, val);
	    return val;
	}

    }

    /**
     * An environment for local values of variables.
     */
    public class LexicalEnv extends AbstractEnv {

	Symbol[] vars;
	Object[] vals;
	LexicalEnv enclosingEnv;

	public LexicalEnv(Symbol[] vars, Object[] vals, LexicalEnv next) {
	    if (vars.length != vals.length)
		throw new Error
		    ("Wrong number of arguments when binding. " +
		     " Expected " + vars.length +
		     " but found " + vals.length);
	    this.vars = vars;
	    this.vals = vals;
	    this.enclosingEnv = next;
	}

	public Object lookup(VarRef ref) {
	    if (ref.up == UNKNOWN)
		findOffsets(ref);
	    if (ref.up == GLOBAL)
		return globalEnv.lookup(ref);
	    LexicalEnv frame = this;
	    for (int up = ref.up; up > 0; up--)
		frame = frame.enclosingEnv;
	    return frame.vals[ref.over];
	}

	public Env bind(Symbol[] vars, Object[] vals) {
	    return new LexicalEnv(vars, vals, this);
	}

	public Object assign(VarRef ref, Object val) {
	    if (ref.up == UNKNOWN)
		findOffsets(ref);
	    if (ref.up == GLOBAL)
		return globalEnv.assign(ref, val);
	    LexicalEnv frame = this;
	    for (int up = ref.up; up > 0; up--)
		frame = frame.enclosingEnv;
	    frame.vals[ref.over] = val;
	    return val;
	}

	protected void findOffsets(VarRef ref) {
	    // Loop so long as we find LexicalEnvs.
	    LexicalEnv frame = this;
	    for (int up = 0; ; up++) {
		int i = frame.findIndex(ref.var);
		if (i >= 0) {
		    ref.up = up;
		    ref.over = i;
		    return;
		}
		frame = frame.enclosingEnv;
		if (frame == null) {
		    ref.up = GLOBAL;
		    return;
		}
	    }
	}

	protected int findIndex(Symbol var) {
	    for (int i = 0; i < vars.length; i++) {
		if (vars[i] == var)
		    return i;
	    }
	    return -1;
	}

    }

}
