/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Sep  8 22:21:43 2005 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, AIAI, University of Edinburgh
 */

package ix.util.ipc;

import java.util.*;
import java.io.*;
import java.net.*;

import ix.util.*;

/**
 * A Connection that is used to send and receive serialized objects
 * via a socket.  N.B. The name of the agent on the other end is known
 * initially only when the connection was created for sending,
 * but in some cases it may be discovered later on.
 */
public class ObjectStreamConnection implements IPC.Connection {

    Object destination;
    Socket sock;
    ObjectInputStream in;
    ObjectOutputStream out;

    public ObjectStreamConnection(Socket s) {
	this("unknown agent", s);
    }

    public ObjectStreamConnection(Object destination, Socket s) {
	this.destination = destination;
	this.sock = s;
	/* /\/: For some reason, messages don't seem to get through
	   if we create the Object streams right away. */
	// in = new ObjectInputStream(sock.getInputStream());
	// out = new ObjectOutputStream(sock.getOutputStream());
	// ? do we need to put a buffer in there ?  /\/
    }

    public Object getDestination() {
	return destination;
    }

    public void setDestination(Object destination) {
	// Check that destination is not already known? /\/
	this.destination = destination;
    }

    public Socket getSocket() {
	return sock;
    }

    public void close() {
	try {
	    sock.close();
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    throw new IPC.IPCException("Problem closing connection", e);
	}
    }

    public synchronized void send(Object contents) {
	// Debug.noteln("Sending to socket", sock);
	try {
	    if (out == null)
		out = new ObjectOutputStream(sock.getOutputStream());
	    out.writeObject(contents);
	    out.flush();
	    // Forget the object(s) just written, so that writing them
	    // again will send their full state instead of just a ref.
	    out.reset();
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    throw new IPC.IPCException
		("Problem sending to connection to " + destination, e);
	}
    }

    public Object receive() {
	try {
	    if (in == null)
		in = new ObjectInputStream(sock.getInputStream());
	    Object result = in.readObject();
	    // Debug.noteln("Received " + result.getClass().getName()
	    //              + " from socket", sock);
	    return result;
	}
	catch (ClassNotFoundException e) {
	    Debug.noteException(e);
	    throw new IPC.IPCException
		("Problem receiving from connection to " + destination, e);
	}
	catch (EOFException e) { 	// a case we interpret
	    Debug.noteException(e);
	    throw new IPC.BrokenConnectionException(this, e);
	}
	catch (SocketException e) {	// another we interpret
	    // In Windows, get "Connection reset by peer ...".
	    Debug.noteException(e);
	    throw new IPC.BrokenConnectionException(this, e);
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    throw new IPC.IPCException
		("Problem receiving from connection to " + destination, e);
	}
    }

}
