/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Aug 27 14:25:23 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.ipc;

import javax.swing.*;

import java.util.*;

import ix.icore.IXAgent;
import ix.icore.Sendable;
import ix.ip2.Ip2;		// shouldn't work only with Ip2 /\/
import ix.ispace.*;
import ix.ispace.event.*;
import ix.iface.util.*;
import ix.util.*;

public class DispatchingStrategy implements ParameterizedCommStrategy {

    IXAgent agent;

    StrategyPanel strategyPanel;

    Strategy defaultStrategy;
    List strategies = new LinkedList();
    Map destinationToStrategyMap = new TreeMap();

    public DispatchingStrategy() {
    }

    public IPC.CommunicationStrategy apply(String[] args) {
	if (args.length < 1)
	    throw new IllegalArgumentException
		("No communication strategies listed for dispatching.");
	for (int i = 0; i < args.length; i++) {
	    String spec = args[i];
	    if (spec.startsWith("default:")) {
		if (defaultStrategy != null)
		    throw new IllegalArgumentException
			("Two defaults specified for dispatching strategy.");
		defaultStrategy =
		    new Strategy
		            (Strings.afterFirst("default:", spec));
		strategies.add(defaultStrategy);
	    }
	    else {
		strategies.add(new Strategy(spec));
	    }
	}
	if (defaultStrategy == null)
	    throw new IllegalArgumentException
		("No default specified for dispatching strategy");
	return this;
    }

    public void sendObject(Object destination, Object contents) {
	Debug.expect(!strategies.isEmpty(), "No strategies for dispatching");

	// /\/: Ensure server setup is sufficiently complete.
	// Perhaps use a ThreadCondition.

	Strategy strategy = ensureStrategy(destination);

	// /\/: Tell the contact-manager about the agent.
	// (Normal sendObject does not do this.)
	// (Assume there are no thread issues ...)
	agent.getContactManager().noteAgent((String)destination);

	// Now send
	strategy.getStrategy()
	    .sendObject(destination, contents);
    }

    public void setupServer(Object destination,
			    IPC.MessageListener listener) {
	Debug.expect(!strategies.isEmpty(), "No strategies for dispatching");

	this.agent = IXAgent.getAgent();
	Debug.expectEquals(destination, agent.getAgentSymbolName(),
			   "Symbol name changed before server setup:");

	// Add some I-Space Tool GUI
	ispaceSetup();

	// Set up all the strategies we'll be using
	// Note that a strategy might change the agent's symbol name.
	// We allow only the first strategy to do this.
	boolean changeAllowed = true;
	for (Iterator i = strategies.iterator(); i.hasNext();) {
	    Strategy s = (Strategy)i.next();
	    Debug.noteln("Calling setupServer for", s);
	    s.setupServer(destination, listener);
	    String agentName = agent.getAgentSymbolName();
	    if (!agentName.equals(destination)) {
		if (changeAllowed)
		    destination = agentName;
		else
		    throw new IllegalStateException
			("Strategy " + s + " attempted to change " +
			 "the agent's symbol name, but only the " +
			 "first-listed strategy is allowed to do so.");
	    }
	    changeAllowed = false;
	}

    }

    synchronized Map getDestinationToStrategyMap() {
	return destinationToStrategyMap;
    }

    synchronized Strategy getStrategy(Object destination) {
	Debug.expect(destination instanceof String);
	return (Strategy)destinationToStrategyMap.get(destination);
    }

    synchronized void setStrategy(Object destination, Strategy s) {
	Debug.expect(destination instanceof String);
	if (s == null)
	    destinationToStrategyMap.remove(destination);
	else
	    destinationToStrategyMap.put(destination, s);
    }

    synchronized Strategy ensureStrategy(Object destination) {
	Strategy s = getStrategy(destination);
	if (s == null) {
	    s = defaultStrategy;
	    setStrategy(destination, s);
	    Debug.noteln("Assigned default strategy " + s + " to " +
			 destination);
	}
	return s;
    }

    synchronized Strategy findStrategy(String name) {
	for (Iterator i = strategies.iterator(); i.hasNext();) {
	    Strategy s = (Strategy)i.next();
	    if (s.getName().equals(name))
		return s;
	}
	return null;
    }

    void usingStrategy(final String agentName, final Strategy s) {
	// Called when handling an incoming message, hence in
	// a variety of threads.  That also means the contact
	// manager will be told of the agent when the message
	// is received by the agent.
	final Strategy before;
	synchronized(this) {
	    before = getStrategy(agentName);
	    if (before != s)
		// Always set the strategy if it's a change.  /\/
		setStrategy(agentName, s);
	    else
		return;
	}
	// Tell the strategy panel
	Util.swingAndWait(new Runnable() {
	    public void run() {
		strategyPanel.nowUsingStrategy(agentName, before, s);
	    }
	});
    }

    class Strategy implements IPC.MessageListener {

	String name;
	IPC.CommunicationStrategy strategy;
	IPC.MessageListener agentsListener;

	Strategy(String name) {
	    this.name = name;
	    this.strategy = IPC.makeCommunicationStrategy(name);
	}

	String getName() {
	    return name;
	}

	IPC.CommunicationStrategy getStrategy() {
	    return strategy;
	}

	public String toString() {
	    return name;
	}

	public void setupServer(Object destination,
				IPC.MessageListener listener) {
	    Debug.expect(agentsListener == null);
	    agentsListener = listener;
	    strategy.setupServer(destination, this);
	}

	public void messageReceived(IPC.InputMessage message) {

	    // /\/: Don't allow messages in until all strategies
	    // have completed their setupSever calls.

	    Object contents = message.getContents();
	    if (contents instanceof Sendable) {
		Sendable sent = (Sendable)contents;
		Name senderName = sent.getSenderId();
		if (senderName == null)
		    // Should be Debug.warn, but it happens too often. /\/
		    Debug.noteln("Strategy " + this + " received" +
				 " a message with no sender-id");
		else
		    // So we know that agent is using this strategy ...
		    usingStrategy(senderName.toString(), this);
	    }
	    // Now let the message be received in the normal way.
	    // This will also let the contact-manager notice that
	    // the sender exists.
	    agentsListener.messageReceived(message);
	}

    }

    void ispaceSetup() {
	// Get the agent's I-Space tool.
	Ip2 ip2 = (Ip2)IXAgent.getAgent();
	ISpaceTool ispace = (ISpaceTool)ip2.ensureTool("I-Space");
	ContactManager contactManager = ip2.getContactManager();

	// Add any agents already known to the contact manager
	// (e.g. from command-line arguments).
	for (Iterator i = contactManager.getAgentData().iterator()
		 ; i.hasNext();) {
	    AgentData data = (AgentData)i.next();
	    ensureStrategy(data.getName());
	}

	// Create the panel to put in the I-Space tool.
	// /\/: Doing it here ensures that any already-known
	// agents noticed above will have a line in the table.
	strategyPanel = new StrategyPanel(ispace);
	contactManager.addContactListener(strategyPanel);

	// Add a tab to the I-Space tool.
	ispace.addTab("Communication", strategyPanel);

    }

    class StrategyPanel extends SelectionPanel implements ContactListener {

	ISpaceTool ispace;
	ContactManager contactManager;

	StrategyPanel(ISpaceTool ispace) {
	    super(ispace.getFrame(), "Agent", "Strategy");
	    this.ispace = ispace;
	    this.contactManager = agent.getContactManager();

	    // /\/: See RelationPanel in ISpaceTool for a comment
	    // on what can't be done in the constructor.
	    load();

	}

	void nowUsingStrategy(String name, Strategy before, Strategy s) {
	    Debug.expect(s != before);
	    if (before != null) {
		Util.displayAndWait
		    (ispace.getFrame(),
		     "Discovered " + name + " using strategy " + s +
		     " instead of " + before);
	    }
	    reload();
	}

	// Define the SelectionPanel abstract methods

	protected ValueComboBox makeValueComboBox() {
	    return new StrategyComboBox();
	}

	protected Map getCurrentValueMap() {
	    return new TreeMap(getDestinationToStrategyMap()); //\/ need copy?
	}

	protected void entryDeleted(String name) {
	    Debug.expect(false, "Delete shouldn't be possible");
	}

	protected void valueChanged(String name, Object value) {
	    Strategy old = getStrategy(name);
	    Debug.expect(old != null, "no old value to change");
	    setStrategy(name, (Strategy)value);
	}

	protected void entryAdded(String name, Object value) {
	    Strategy old = getStrategy(name);
	    if (old != null)
		throw new IllegalArgumentException
		    ("Agent " + name + " is already in the table");
	    setStrategy(name, (Strategy)value);
	    contactManager.noteAgent(name);
	}

	// ContactListener methods

	public void contactChange(ContactEvent e) {
	    if (e.isNewContact()) {
		String name = e.getNewData().getName();
		ensureStrategy(name);
	    }
	    else if (e.isDeletion()) {
		String name = e.getOldData().getName();
		setStrategy(name, null); // == remove(name)
	    }
	    // /\/: For now just reload.  Eventually need to figure
	    // out just what has to change and warn the user if it affects
	    // any uncommitted changes made in the panel.
	    this.reload();
	}

    }

    /** A combo-box for selecting communication strategies  */
    class StrategyComboBox extends ValueComboBox {

	public StrategyComboBox() {
	    super();
	    for (Iterator i = strategies.iterator(); i.hasNext();) {
		Strategy s = (Strategy)i.next();
		addItem(s);
	    }
	    setSelectedValue(defaultStrategy);
	}

	public Object getSelectedValue() {
	    return (Strategy)getSelectedItem();
	}

	public void setSelectedValue(Object item) {
	    Strategy s = (Strategy)item;
	    setSelectedItem(s);
	}

    }

}
