/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Aug  8 19:39:09 2008 by Jeff Dalton
 * Copyright: (c) 2001, 2004, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.util.context;

import java.util.*;

import ix.util.*;

/**
 * A context-holding strategy that associates context-holders with
 * threads.  The holder for a thread is created by the {@link #initialHolder()}
 * method.  (In some cases, it may be better to override that method to
 * have it return null, so that the holder must be explicitly set.)
 * More than one thread may be given the same context-holder, however
 * this must be done with care, because some objects that use contexts
 * frequently change their current context, and two threads doing that
 * with the same holder could confuse each other.
 */
public class ThreadLocalHoldingStrategy extends ContextHoldingStrategy {

    protected ThreadLocal<ContextHolder> threadContextHolder =
	new ThreadLocal<ContextHolder>() {
	    protected ContextHolder initialValue() {
		return initialHolder();
	    }
        };

    public ThreadLocalHoldingStrategy() {
    }

    /**
     * Called to create a holder for a thread that doesn't already have one.
     * Unless overridden in a subclass, this method returns an instance of
     * {@link BasicContextHolder}.
     */
    protected ContextHolder initialHolder() {
	Debug.noteln("Creating context holder for", Thread.currentThread());
	return new BasicContextHolder();
    }

    /**
     * Returns the ContextHolder for the current thread.
     */
    public ContextHolder getContextHolder() {
	return threadContextHolder.get();
    }

    /**
     * Sets the ContextHolder for the current thread.
     */
    public void setContextHolder(ContextHolder h) {
	threadContextHolder.set(h);
    }

    /**
     * Clears contexts in the current thread's holder by telling it
     * to set its current context to the root context.
     */
    public void clearContexts() {
	threadContextHolder.get().setContext(Context.rootContext);
    }

}
