/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 23 00:24:14 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.util.context;

import java.lang.reflect.Proxy;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import java.lang.reflect.InvocationTargetException;

import java.util.*;

import ix.util.*;

/**
 * Provides a static method that can be used to wrap a context-managing
 * proxy around an object.
 */
public class ContextLockingProxy {

    private ContextLockingProxy() {}

    /**
     * Creates a proxy around the specified instance of the interface.
     * The invocation handler will be a {@link ContextLockingProxy.Handler},
     * and the handler's where everything interesting happens.
     */
    public static <T> T newProxyInstance(Class<T> interfaceClass,
					 Object instance) {
	return (T)Proxy.newProxyInstance(interfaceClass.getClassLoader(),
					 new Class[] { interfaceClass },
					 new Handler(instance));
    }

    /**
     * The invocation-handler for a context-locking proxy.
     * When the handler is created, it grabs the current context
     * which is assumed to be the right context for the first method
     * call handled by the proxy.  That's saved as the "instance context".
     * Then, whenever the handler is given a method to call, it
     * does the following while synchronized on the current
     * {@link ContextHolder}:
     * <ol>
     * <li>Remember the current context.
     * <li>Set the current context to the saved instance context.
     * <li>Invoke the method, and then in a "finally" clause:
     * <li>Record as the instance context whatever context is current
     *     now that the method's returned, and
     * <li>restore the context remembered earlier.
     * </ol>
     */
    public static class Handler implements InvocationHandler {

	protected Object instance;

	protected Context instanceContext = Context.getContext();

	public Handler(Object instance) {
	    this.instance = instance;
	}

	public Object invoke(Object proxy, Method method, Object[] args)
	       throws Throwable {
	    ContextHolder holder = Context.getContextHolder();
	    synchronized (holder) {
		Context savedContext = holder.getContext();
		try {
		    holder.setContext(instanceContext);
		    return method.invoke(instance, args);
		}
		catch (InvocationTargetException e) {
		    throw e.getCause();
		}
		finally {
		    instanceContext = holder.getContext();
		    holder.setContext(savedContext);
		}
	    }
	}

    }

}
