/* File: Parameters.java
 * Contains: Methods for command-line arguments and Applet parameters
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: February 1998
 * Updated: Fri Sep 25 16:05:36 2009 by Jeff Dalton
 * Copyright: (c) 1998 - 2009, AIAI, University of Edinburgh
 */

package ix.util;

import java.applet.Applet;
import java.awt.Dimension;

import java.util.*;
import java.io.*;
import java.net.URL;		// for "-load"

import ix.util.xml.XML;		// for "-load"

/**
 * The Parameters class contains methods that allow information from
 * application command-line arguments and from Applet parameters to
 * be used in a uniform way.  The class cannot be instantiated.
 *
 * @see ParameterException
 */
public abstract class Parameters {

    private static boolean isApplet = false;
    private static boolean isInteractive = true;

    private static Applet applet;

    private static AccessRecordingProperties parameters
        = new AccessRecordingProperties();

    /**
     * Sets a parameter value.  After recording the value, it
     * calls {@link Debug#processParameter(String, String)},
     * in case the paraemter affects debugging output.
     * The setParameter method doesn't produce any debugging
     * output of its own until after that call.
     */
    public static synchronized void setParameter(String pname, String value) {
	// Set first so that Debug.processParameter can use
	// the Parameters class's "get" methods.
	parameters.put(pname, value);
	Debug.processParameter(pname, value);
	if (Debug.on)
	    Debug.noteln("Set parameter \"" + pname + "\" = " +
			 "\"" + value + "\"");
    }

    public static synchronized Properties getParameters() {
	return parameters;
    }

    public static synchronized boolean isInteractive() {
	return isInteractive;
    }
    public static synchronized void setIsInteractive(boolean v) {
	Debug.noteln("Setting interactive = " + v);
	isInteractive = v;
    }

    /**
     * Returns the value of the parameter if one was specified,
     * else returns null.
     */
    public static synchronized String getParameter(String pname) {
	String v = parameters.getProperty(pname);
        if (v != null) {
            return v;
        }
        else if (applet != null) {
            try {
                v = applet.getParameter(pname);
            }
            catch (NullPointerException e) {
                Debug.noteln("Couldn't get applet parameter " + pname);
                return null;
            }
            if (v != null) {
                // Store in main table and make sure access is recorded.
                setParameter(pname, v);
                return parameters.getProperty(v);
            }
            return v;
        }
        else {
            return null;
        }
    }

    public static String getParameter(String pname, String defaultValue) {
        String v = getParameter(pname);
        return v != null ? v : defaultValue;
    }

    public static boolean haveParameter(String pname) {
	return getParameter(pname) != null;
    }

    public static String requireParameter(String pname) {
	String v = getParameter(pname);
	if (v == null)
	    throw new ParameterException
		("The required parameter " + Strings.quote(pname) +
                 " has not been supplied.");
	else
	    return v;
    }

    /**
     * Returns true if the haveParameter method or any of the
     * parameter "get" methods has been called on the indicated
     * parameter name; else returns false.
     */
    public static synchronized boolean usedParameter(String pname) {
	return parameters.accessedProperty(pname);
    }

    public static int getInt(String pname) {
	String value = getParameter(pname);
	if (value == null)
	    throw bogusParameter("int", pname, "null");
	else {
	    try {
		// N.B. decode(value).intValue() rather than parseInt(value).
		// This allows 0x and # to indicate hex and 0 octal.
		return Integer.decode(value).intValue();
	    }
	    catch (NumberFormatException e) {
		Debug.noteException(e);
		throw bogusParameter("int", pname, value);
	    }
	}
    }

    public static int getInt(String pname, int defaultValue) {
	if (haveParameter(pname))
	    return getInt(pname);
	else
	    return defaultValue;
    }

    public static int getInt(String pname, int radix, int defaultValue) {
	String value = getParameter(pname);
	if (value == null)
	    return defaultValue;
	else {
	    try {
		return Integer.parseInt(value, radix);
	    }
	    catch (NumberFormatException e) {
		Debug.noteException(e);
		throw bogusParameter("int", pname, value);
	    }
	}
    }

    public static boolean getBoolean(String pname) {
	return getBoolean(pname, false);
    }

    public static boolean getBoolean(String pname, boolean defaultValue) {
	String value = getParameter(pname);
	if (value == null)
	    return defaultValue;
	else if (value.equals("")) 	// -pname without =value
	    return true;
	else if (value.equals("true"))
	    return true;
	else if (value.equals("false"))
	    return false;
	else
	    throw bogusParameter("boolean", pname, value);
    }

    public static Dimension getDimension(String pname) {
	String value = getParameter(pname);
	if (value == null)
	    return null;

	String[] parts = Strings.breakAtFirst("x", value);
	try {
	    int width = Integer.parseInt(parts[0]);
	    int height = Integer.parseInt(parts[1]);
	    return new Dimension(width, height);
	}
	catch (Exception e) {
	    // Typically a NumberFormatException.
	    Debug.noteException(e);
	    throw bogusParameter("Dimension", pname, value);
	}
    }

    public static List getList(String pname) {
	return getList(pname, Collections.EMPTY_LIST);
    }

    public static List getList(String pname, List defaultValue) {
	String value = getParameter(pname);
	if (value == null)
	    return defaultValue;
	else
	    return Strings.breakAt(",", value);
    }

    public static Class getClass(String pname) {
	return getClass(pname, null);
    }

    public static Class getClass(String pname, Class defaultValue) {
	String value = getParameter(pname);
	if (value == null)
	    return defaultValue;
	else try {
	    return Class.forName(value);
	}
	catch (ClassNotFoundException e) {
	    Debug.noteException(e);
	    throw bogusParameter("Class", pname, value);
	}
    }

    private static ParameterException bogusParameter(
                     String type, String pname, String value) {
	return new ParameterException("Parameter \"" + pname + "\" " +
				    "has a non-" + type + " value: " +
				    value);
    }

    public static synchronized void checkParameterUse() {
	if (!allParametersWereUsed()) {
	    Vector unused = new Vector();
	    for (Enumeration keys = parameters.propertyNames();
		 keys.hasMoreElements(); ) {
		String name = (String)keys.nextElement();
		if (!parameters.accessedProperty(name))
		    unused.addElement(name);
	    }
	    Debug.expect(!unused.isEmpty());
	    Debug.noteln("");
	    Debug.noteln
		("Warning: The following parameters have not yet been used:");
	    Debug.noteElements(unused.elements(), "   ");
	    Debug.noteln("");
	}
    }

    public static synchronized boolean allParametersWereUsed() {
	for (Enumeration keys = parameters.propertyNames();
	     keys.hasMoreElements(); ) {
	    if (!parameters.accessedProperty((String)keys.nextElement()))
		return false;
	}
	return true;
    }

    /* *
     * * Methods for command-line arguments
     * */

    /**
     * Parse a String[] of command-line arguments.  The syntax of an
     * argument is -name=value.  The value assigned to a name can the
     * be obtained as a String by calling getParameter("name").  Other
     * get-methods can return values of other types.<p>
     *
     * If no value is given, the value is the empty string "".
     * For getBoolean(name), this is equivalent to "true".<p>
     *
     * The syntax -not name, or -no name, is equivalent to -name=false.<p>
     *
     * The syntax -load resource-name can be used to read parameter values
     * from a file, from a URL, or from a resource looked up by a class-
     * loader.  The contents should be lines in name=value syntax.
     */
    public static void processCommandLineArguments(String[] argv) {
	// boolean errors = false;
	Enumeration args = Seq.elements(argv);
	while (args.hasMoreElements()) {
	    String arg = (String)args.nextElement();
	    String[] part = Strings.breakAtFirst("=", arg);
	    String pname = part[0], value = part[1];
	    if (!pname.startsWith("-"))
		complain(pname, "does not start with \"-\"");
	    else
		pname = pname.substring(1);
	    if (pname.startsWith("n") &&
		(pname.equals("not") || pname.equals("no"))) {
		// Handle -not pname and -no pname which set to "false"
		if (!value.equals(""))
		    complain(arg, "has " + pname + " with =");
		else if (args.hasMoreElements()) {
		    pname = (String)args.nextElement();
		    setParameter(pname, "false");
		}
		else
		    complain(pname, "was not followed by a parameter name");
	    }
	    else if (pname.equals("load")) {
		// handle -load filename
		if (!value.equals(""))
		    complain(arg, "has " + pname + " with =");
		else if (args.hasMoreElements()) {
		    try { loadParameters((String)args.nextElement()); }
		    catch (Exception e) {
			Debug.displayException(e);
			if (!isInteractive())
			    throw new RethrownException(e);
		    }
		}
		else
		    complain(pname, "was not followed by a resource name");
	    }
	    else 
		// An ordinary pname=value parameter
		setParameter(pname, value);
	}

    }

    private static void complain(String pname, String message) {
	Debug.warn("Parameter \"" + pname + "\" " + message);
    }

    public static synchronized void loadParameters(String resourceName) {
	Properties props = new Properties();
	URL url = XML.toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException
		("Can't find a property file named " +
		 Strings.quote(resourceName));
	try {
	    props.load(url.openStream());
	}
	catch (IOException e) {
	    throw new RethrownException(e);
	}
	for (Enumeration keys = props.propertyNames();
	     keys.hasMoreElements(); ) {
	    String pname = (String)keys.nextElement();
	    setParameter(pname, (String)props.get(pname));
	}
    }


    /* *
     * * Methods for Applet parameters
     * */

    public static synchronized boolean isApplet() {
	return isApplet;
    }
    public static synchronized void setApplet(Applet a) {
        applet = a;
	isApplet = (applet != null);
    }


    /* *
     * * Utilities
     * */

    static class AccessRecordingProperties extends Properties {

	private Map<String,Boolean> accessTable =
            new HashMap<String,Boolean>();

	public AccessRecordingProperties() {
	    super();
	}

	public synchronized String getProperty(String key) {
	    accessTable.put(key, Boolean.TRUE);
	    return super.getProperty(key);
	}

	public synchronized String getProperty(String key,
					       String defaultValue) {
	    accessTable.put(key, Boolean.TRUE);
	    return super.getProperty(key, defaultValue);
	}

	public synchronized boolean accessedProperty(String key) {
	    return accessTable.get(key) != null;
	}

    }

}
