/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 22 16:34:44 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.net.*;
import java.util.*;

import ix.util.xml.*;
import ix.util.lisp.*;

/**
 * Maps both ways between short and long names.
 */
public class NameMapper {

    Gensym.Generator gensym = new Gensym.Generator();

    protected Map longToShort = new HashMap();
    protected Map shortToLong = new HashMap();

    public NameMapper() {
    }

    public NameMapper(Map shortToLong) {
	this.shortToLong = new HashMap(shortToLong);
	this.longToShort = Collect.transposeMap(this.shortToLong);
    }

    public boolean hasShortName(String longName) {
	return longToShort.containsKey(longName);
    }

    public String shortName(String longName) {
	String s = (String)longToShort.get(longName);
	if (s == null)
	    throw new IllegalArgumentException
		("Unknown long name " + Strings.quote(longName));
	else
	    return s;		 
    }

    /*
    public Symbol sortSymbol(String longName) {
	return Symbol.intern(shortName(longName));
    }
    */

    public boolean hasLongName(String shortName) {
	return shortToLong.containsKey(shortName);
    }

    public String longName(String shortName) {
	String l = (String)shortToLong.get(shortName);
	if (l == null)
	    throw new IllegalArgumentException
		("Unknown short name " + Strings.quote(shortName));
	else
	    return l;
    }

    public Map getShortToLongMap() {
	return Collections.unmodifiableMap(shortToLong);
    }

    public Map getLongToShortMap() {
	return Collections.unmodifiableMap(longToShort);
    }

    public void setMapping(String longName, String shortName) {
	longToShort.put(longName, shortName);
	shortToLong.put(shortName, longName);
    }

    public String mapLongName(Object obj) { // for URIs etc
	return mapLongName(obj.toString());
    }

    /**
     * Returns a suitable short name for a long name, generating
     * and storing a short name if one is not already in the map.
     */
    public String mapLongName(String longName) {
	String s = (String)longToShort.get(longName);
	if (s == null) {
	    s = generateShortName(longName);
	    setMapping(longName, s);
	}
	Debug.noteln("Short name for " + Strings.quote(longName) +
		     " = " + Strings.quote(s));
	return s;
    }

    public String generateShortName(String longName) {
	// Remember that the short name needs to be unique
	// within this mapper so that we can get the long name
	// back from the short name.
	String s = plausibleShortName(longName);
	String exists = (String)shortToLong.get(s);
	if (exists == null)
	    return s;
	else {
	    // Oops!  There's already a long name for the plausible short name.
	    Debug.expect(!exists.equals(longName),
			 "shouldn't have needed a short name for", longName);
	    s = gensym.nextString(s);
	    Debug.expect(s.length() > 0,
			 "empty plausible name for", longName);
	    Debug.expect(shortToLong.get(s) == null,
			 "generated string is not unique", s);
	    return s;
	}
    }

    public String plausibleShortName(String longName) {
	if (hasSeparator(longName, '#')) {
	    return Strings.afterLast("#", longName);
	}
	else if (hasSeparator(longName, '/')) {
	    String s = Strings.afterLast("/", longName);
	    return Strings.beforeLast(".", s);
	}
	else if (hasSeparator(longName, '\\')) {
	    String s = Strings.afterLast("\\", longName);
	    return Strings.beforeLast(".", s);
	}
	else
	    return longName;	// /\/ for now
    }

    protected boolean hasSeparator(String s, int sepChar) {
	// True of the string contains the separator and
	// the separator is not at the end of the string.
	int i = s.indexOf(sepChar);
	return i >= 0 && i < s.length() - 1;
    }

}
