/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Nov 11 23:19:08 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import ix.util.lisp.*;

/**
 * Static graph utilities.
 */
public final class Graphs {

    private Graphs() { }	// block instantiation

    /**
     * Makes a directed graph from a map that maps each vertex
     * to its successors.
     *
     * @see #toMap(Map, DirectedGraph)
     */
    public static DirectedGraph makeDirectedGraph(final Map map) {
	return new DirectedGraph() {
	    public Collection getAllNodes() {
		return map.keySet();
	    }
	    public Collection getRoots() {
		return map.keySet(); // /\/
	    }
	    public Collection getSuccessors(Object node) {
		return (Collection)map.get(node);
	    }
	    public boolean isBefore(Object a, Object b) {
		return getSuccessors(a).contains(b);
	    }
	};
    }

    /**
     * Makes a directed graph from a collection of root vertices
     * and a function that maps each vertex to its successors.
     */
    public static DirectedGraph makeDirectedGraph(final Collection roots,
						  final Function1 f) {
	return new DirectedGraph() {
	    Collection allNodes = null;
	    public Collection getAllNodes() {
		if (allNodes == null)
		    allNodes = topologicalSort(this);
		return allNodes;
	    }
	    public Collection getRoots() {
		return roots;
	    }
	    public Collection getSuccessors(Object node) {
		return (Collection)f.funcall(node);
	    }
	    public boolean isBefore(Object a, Object b) {
		return getSuccessors(a).contains(b);
	    }
	};
    }

    /**
     * Extends a map by adding a mapping from each vertex in a
     * directed graph to the vertex's successors in that graph.
     *
     * @return the modified map.
     *
     * @see #makeDirectedGraph(Map)
     */
    public static Map toMap(Map m, DirectedGraph g) {
	for (Iterator i = g.getAllNodes().iterator(); i.hasNext();) {
	    Object vertex = i.next();
	    m.put(vertex, g.getSuccessors(vertex));
	}
	return m;
    }

    /**
     * Returns a set containing the vertices that have no predecessors.
     */
    public static Set minimalElements(DirectedGraph g) {
	Collection all = g.getAllNodes();
	// Start with all of the vertices and remove every one
	// that appears as a successor.
	Set result = new HashSet(all);
	for (Iterator i = all.iterator(); i.hasNext();) {
	    Object v = i.next();
	    result.removeAll(g.getSuccessors(v));
	}
	return result;
    }

    /**
     * Returns a set containing the vertices that have no successors.
     */
    public static Set maximalElements(DirectedGraph g) {
	Set result = new HashSet();
	for (Iterator i = g.getAllNodes().iterator(); i.hasNext();) {
	    Object v = i.next();
	    Collection ss = g.getSuccessors(v);
	    if (ss == null || ss.isEmpty()) // do we need the null test? /\/
		result.add(v);
	}
	return result;
    }

    /**
     * Returns a graph in which each vertex has an edge back to itself.
     */
    public static DirectedGraph makeReflexive(DirectedGraph g) {
	Map m = new StableHashMap();
	for (Iterator i = g.getAllNodes().iterator(); i.hasNext();) {
	    Object n = i.next();
	    Collection successors = g.getSuccessors(n);
	    m.put(n, Collect.union(successors, Lisp.list(n)));
	}
	// /\/: Loses track of which nodes are roots.
	return makeDirectedGraph(m);
    }

    /**
     * Returns a graph in which vertex-to-successor edges go in the
     * opposite direction from those in the original graph.
     */
    public static DirectedGraph transpose(DirectedGraph g) {
	// We'll build a Map to respresent the transposed graph.
	Map m = new StableHashMap();
	Collection all = g.getAllNodes();
	for (Iterator i = all.iterator(); i.hasNext();) {
	    Object v = i.next();
	    for (Iterator j = g.getSuccessors(v).iterator(); j.hasNext();) {
		Object s = j.next();
		// We have v -> s and want s -> v.
		// If each of v's successors appears only once in
		// v's successor collection, then we'll try to add
		// v only once to s's successors in the transposed
		// graph.  Therefore we can use a list.
		List ss = (List)m.get(s);
		if (ss == null) {
		    ss = new LinkedList();
		    m.put(s, ss);
		}
		ss.add(v);
	    }
	}
	// Any vertices that were not successors in the original
	// graph will not yet have an entry in the transposed graph.
	for (Iterator i = all.iterator(); i.hasNext();) {
	    Object v = i.next();
	    List vs = (List)m.get(v);
	    if (vs == null)
		m.put(v, new ArrayList(0));
	}
	return makeDirectedGraph(m);
    }

    /**
     * DAG topological sort, returning ancestors before descendents.
     *
     * @see TopologicalSorter
     */
    public static List topologicalSort(final DirectedGraph g) {
	return new TopologicalSorter() {
	    protected Collection getChildren(Object node) {
		return g.getSuccessors(node);
	    }
	}.sort(g.getRoots());
    }

    /**
     * Transitive closure of a directed acyclic graph (DAG).
     *
     * @see #transitiveClosure(DirectedGraph)
     * @see DAGTransitiveClosure
     */
    public static DAGTransitiveClosure dagTransitiveClosure(DirectedGraph g) {
	return new DAGTransitiveClosure(g);
    }

    /**
     * Transitive closure of a relation.
     * The graph is allowed to contain cycles.
     *
     * @see #dagTransitiveClosure(DirectedGraph)
     * @see FullTransitiveClosure
     */
    public static FullTransitiveClosure transitiveClosure(DirectedGraph g) {
	return new FullTransitiveClosure(g);
    }

    /**
     * DAG Longest-path-lengths algorithm.
     *
     * <p>Finds distances from the root vertices by finding the number of
     * steps in the longest path along successor links from a roots without
     * predecessors to each vertex, where the length of each link is 1.
     * (The term "root" implies no predecessors, but the algorithm
     * begins with a topological sort that should straighten things
     * out if a root happens to have some.)
     *
     * <p>The results are returned in a Map that maps vertices to
     * distances.  Vertices must be represented by objects that
     * are equals-unique.  The distance values are represented
     * by {@link IntRef}s.
     *
     * <p>This is O(v+e) where v is the number of vertices and e the number
     * of edges (links to successors).
     *
     * <p><i>Translated from the O-Plan TF compiler's graph utilities.</i>
     */
    public static Map findLongestPathLengths(DirectedGraph g) {
	Map distances = new HashMap();
	List vertexSequence = topologicalSort(g);
	for (Iterator i = vertexSequence.iterator(); i.hasNext();) {
	    distances.put(i.next(), new IntRef(0));
	}
	for (Iterator i = vertexSequence.iterator(); i.hasNext();) {
	    Object v = i.next();
	    IntRef distanceToV = (IntRef)distances.get(v);
	    int minDistanceFromV = distanceToV.get() + 1;
	    for (Iterator j = g.getSuccessors(v).iterator(); j.hasNext();) {
		Object child = j.next();
		IntRef distanceToChild = (IntRef)distances.get(child);
		if (minDistanceFromV > distanceToChild.get())
		    distanceToChild.set(minDistanceFromV);
	    }
	}
	return distances;
    }

    /**
     * Makes a BoundedGraph from a DirectedGraph by adding
     * start and finish nodes.  The start and finish will be
     * plain Objects.  Note that the start and finish are
     * always added, even if the graph already contains a
     * unique minimal or maximal node.
     */
    public static BoundedGraph makeBoundedGraph(DirectedGraph g) {
	return makeBoundedGraph(g, new Function0() {
	    public Object funcall() { return new Object(); }});
    }
	
    /**
     * Makes a BoundedGraph from a DirectedGraph by adding
     * start and finish nodes.  The start and finish are created
     * by calling the nodeFactory.  Note that the start and finish
     * are always added, even if the graph already contains a unique
     * minimal or maximal node.
     */
    public static BoundedGraph makeBoundedGraph(final DirectedGraph g,
						Function0 nodeFactory) {
	final Object start = nodeFactory.funcall();
	final Object finish = nodeFactory.funcall();
	return new BoundedGraph() {
	    public Object getStartNode() { return start; }
	    public Object getFinishNode() { return finish; }
	    public Collection getAllNodes() {
		LList g_nodes = LList.newLList(g.getAllNodes());
		return new Cons(start, new Cons(finish, g_nodes));
	    }
	    public Collection getRoots() { return Lisp.list(start); }
	    public Collection getSuccessors(Object node) {
		if (node == start) {
		    Collection g_succ = g.getRoots();
		    return g_succ.isEmpty() ? Lisp.list(finish) : g_succ;
		}
		else if (node == finish)
		    return Lisp.NIL;
		else {
		    Collection s = g.getSuccessors(node);
		    // /\/: Should return a collection of the same
		    // class as s.
		    return s.isEmpty() ? Lisp.list(finish) : s;
		}
	    }
	    public boolean isBefore(Object a, Object b) {
		return getSuccessors(a).contains(b); // inefficient /\/
	    }
	    public String toString() {
		return "BoundedGraph[" + g + "]";
	    }
	};
    }

}
