/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 19 01:57:55 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

/**
 * The transitive closure of a directed acyclic graph (DAG).
 */
public class DAGTransitiveClosure implements TransitiveClosure {

    DirectedGraph baseGraph;

    Map successorMap = new HashMap(); // vertex -> Set of vertices

    public DAGTransitiveClosure(DirectedGraph g) {
	this.baseGraph = g;
	computeTransitiveClosure();
    }

    /*
     * DirectedGraph methods
     */

    public Collection getAllNodes() {
	return baseGraph.getAllNodes();
    }

    public Collection getRoots() {
	return baseGraph.getRoots();
    }

    public Collection getSuccessors(Object node) {
	return (Set)successorMap.get(node);
    }

    public boolean isBefore(Object v1, Object v2) {
	return getSuccessors(v1).contains(v2);
    }

    /*
     * The algorithm
     */

    protected Object START_MARK = new Object() {
	public String toString() {
	    return "#<TransitiveClosure START_MARK>";
	}
    };

    protected void computeTransitiveClosure() {
	// Computes the closure while doing a topological sort.
	walk(baseGraph.getRoots());
    }

    protected void walk(Collection items) {
	for (Iterator i = items.iterator(); i.hasNext();) {
	    walkVertex(i.next());
	}
    }

    protected void walkVertex(Object at) {
	Object value = successorMap.get(at);
	if (value == START_MARK)
	    throw new ConsistencyException("Cycle at", at);
	else if (value instanceof Set)
	    ;		// already visited
	else {
	    successorMap.put(at, START_MARK);
	    Collection successors = baseGraph.getSuccessors(at);
	    walk(successors);
	    // Now every vertex later in the topological sort
	    // should already have the correct value, and we can
	    // use those values when computing the value for
	    // the current vertex, "at".
	    Set result = new HashSet();
	    for (Iterator i = successors.iterator(); i.hasNext();) {
		Object s = i.next();
		Set s_value = (Set)successorMap.get(s);
		result.add(s);
		result.addAll(s_value);
	    }
	    successorMap.put(at, result);
	}
    }

}
