/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Nov 22 17:47:55 2007 by Jeff Dalton
 * Copyright: (c) 2005, 2007, AIAI, University of Edinburgh
 */

package ix.test.xml;

import java.util.*;

import ix.icore.Variable;
import ix.icore.plan.Plan;
import ix.icore.domain.Domain;
import ix.icore.domain.End;
import ix.icore.process.PNodeEnd;

import ix.iplan.AutoTester;
import ix.iplan.PlanTest;
import ix.iplan.PlanCheckingSimulator;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Describes a missionary and cannibals plan test.
 */
public class MissionaryTest extends PlanTest {

    static final Symbol MOVE = Symbol.intern("move");
    static final Symbol LEFT = Symbol.intern("left");
    static final Symbol RIGHT = Symbol.intern("right");

    public MissionaryTest() {
	super();
    }

    /**
     * Returns a TestRunner that runs an extra simulation.
     *
     * @see MissionaryTestRunner
     */
    public AutoTester.TestRunner makeTestRunner(AutoTester auto) {
	return new MissionaryTestRunner(this, auto);
    }

    protected static class MissionaryTestRunner extends AutoTester.TestRunner {

	AutoTester auto;
	MissionaryTest test;

	MissionaryTestRunner(MissionaryTest test, AutoTester auto) {
	    auto.super(test);
	    this.auto = auto;
	    this.test = test;
	}

	/**
	 * Runs the usual simulation using the PlanCheckingSimulator,
	 * then runs a second simulation that uses the same linearization
	 * of the plan but considers only the <tt>move</tt> activities.
	 * The moves are checked to ensure that they produced a connected
	 * series of state changes that reaches the goal state.
	 */
	protected Map runSimulation(PlanCheckingSimulator sim) {
	    Map result = super.runSimulation(sim);
	    simulate(sim.getExecutionOrder());
	    return result;
	}

	void simulate(List nodeEnds) {
	    LList state = Lisp.elementsFromString("left 3 3 0 0");
	    LList goal = Lisp.elementsFromString("right 0 0 3 3");
	    for (Iterator i = nodeEnds.iterator(); i.hasNext();) {
		PNodeEnd ne = (PNodeEnd)i.next();
		if (ne.getEnd() != End.BEGIN)
		    continue;
		LList pattern = ne.getNode().getPattern();
		if (pattern.get(0) != MOVE)
		    continue;
		pattern = (LList)Variable.removeVars(pattern);
		LList moveState = (LList)pattern.get(1);
		LList moveMove = (LList)pattern.get(2);
		if (!moveState.equals(state))
		    auto.testFailure(test, ne + " doesn't move from " + state);
		state = nextState(state, moveMove);
	    }
	    auto.traceln("Reaches state", state);
	    if (!state.equals(goal))
		auto.testFailure(test, "Failed to reach goal state.");
	}

	LList nextState(LList state, LList move) {
	    Symbol bank = (Symbol)state.get(0);
	    int ml = ((Long)state.get(1)).intValue();
	    int cl = ((Long)state.get(2)).intValue();
	    int mr = ((Long)state.get(3)).intValue();
	    int cr = ((Long)state.get(4)).intValue();
	    int m = ((Long)move.get(0)).intValue();
	    int c = ((Long)move.get(1)).intValue();
	    if (bank == LEFT) {
		bank = RIGHT; ml -= m; cl -= c; mr += m; cr += c;
	    }
	    else if (bank == RIGHT) {
		bank = LEFT; ml += m; cl += c; mr -= m; cr -= c;
	    }
	    else {
		auto.testFailure(test, "invalid state " + state);
	    }
	    return Lisp.list(bank, new Long(ml), new Long(cl),
			     new Long(mr), new Long(cr));
	    
	}

    }

}
