/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed May 28 01:14:03 2008 by Jeff Dalton
 * Copyright: (c) 2008, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;

import ix.icore.process.PNodeEnd;
import ix.icore.domain.End;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A simple block-stacking simulator for blocks represented as lists
 * of {@link Symbol}s.
 */
public class SimpleBlockSim {

    LList initialTowers;
    LList finalTowers;
    LList towers;

    public SimpleBlockSim(List initialTowers, List finalTowers) {
	this.initialTowers = LList.LListify(initialTowers);
	this.finalTowers = LList.LListify(finalTowers);
	this.towers = this.initialTowers;
    }

    public SimpleBlockSim(BlockStacker stacker) {
	this(stacker.getInitialTowers(), stacker.getFinalTowers());
    }

    public LList getTowers() {
	return towers;
    }

    public boolean isFinished() {
	return Collect.equalAsSets(towers, finalTowers);
    }

    public void simulatePlan(List<PNodeEnd> nodeEnds) {
	for (PNodeEnd ne: nodeEnds) {
	    if (ne.getEnd() != End.BEGIN)
		continue;
	    LList pattern = ne.getNode().getPattern();
	    String verb = pattern.get(0).toString();
	    if (verb.equals("move")) {
		Symbol block = Util.mustBe(Symbol.class, pattern.get(1));
		Symbol to = Util.mustBe(Symbol.class, pattern.get(2));
		moveBlock(block, to);
	    }
	    else if (verb.equals("move-to-table")) {
		Symbol block = Util.mustBe(Symbol.class, pattern.get(1));
		moveToTable(block);
	    }
	}
    }

    public void moveBlock(Symbol block, Symbol to) {
	LListCollector nextTowers = new LListCollector();
	boolean foundBlock = false;
	boolean foundTo = false;
	for (Iterator i = towers.iterator(); i.hasNext();) {
	    LList tower = (LList)i.next();
	    Symbol blk = (Symbol)tower.car();
	    if (blk == block) {
		// Take block off tower
		foundBlock = true;
		LList tow = tower.cdr();
		if (!tow.isEmpty())
		    nextTowers.add(tow);
	    }
	    else if (blk == to) {
		// Put block on tower
		foundTo = true;
		nextTowers.add(new Cons(block, tower));
	    }
	    else
		nextTowers.add(tower);
	}
	if (! (foundBlock && foundTo))
	    throw new IllegalStateException
		("Could not find" +
		 (foundBlock ? " " : " from " + block) +
		 (foundTo ? "" : " to " + to));
	towers = nextTowers.contents();
    }

    public void moveToTable(Symbol block) {
	LListCollector nextTowers = new LListCollector();
	boolean foundBlock = false;
	for (Iterator i = towers.iterator(); i.hasNext();) {
	    LList tower = (LList)i.next();
	    Symbol blk = (Symbol)tower.car();
	    if (blk == block) {
		// Take block off tower
		foundBlock = true;
		LList tow = tower.cdr();
		if (!tow.isEmpty())
		    nextTowers.add(tow);
		// And give it its own tower, ie put it on the table
		nextTowers.add(Lisp.list(block));
	    }
	    else
		nextTowers.add(tower);
	}
	if (!foundBlock)
	    throw new IllegalStateException
		("Could not find " + block + " to move to table");
	towers = nextTowers.contents();
    }

}


