/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep  9 17:49:42 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;

import ix.ip2.*;
import ix.icore.Status;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Constrains variables by matching a pattern against a choice of values.
 */
public class MatchChoiceHandler extends ActivityHandler {

    static final Symbol S_MATCH_CHOICE = Symbol.intern("match-choice");

    static final LList activityPattern =
	(LList)Lisp.readFromString("(match-choice ?pattern = &rest ?values)");

    Ip2 ip2;

    public MatchChoiceHandler(Ip2 ip2) {
	super("Match against choices");
	this.ip2 = ip2;
    }

    public List getSyntaxList() {
	return (LList)Lisp.readFromString
	    ("((match-choice ?pattern = ?value ...))");
    }

    public boolean appliesTo(AgendaItem item) {
	LList pattern = item.getPattern();
	return SimpleMatcher.match(activityPattern, pattern) != null;
    }

    public void addHandlerActions(AgendaItem item) {
	// appliesTo will have already been called.
	item.addAction(new MatchChoiceAction(item));
    }

    class MatchChoiceAction extends HandlerAction {

	AgendaItem item;

	MatchChoiceAction(AgendaItem item) {
	    this.item = item;
	    this.shortDescription = "Take Choices";
	}

	public boolean isReady() {
	    return true;
	}

	public void handle(AgendaItem item) {
	    // Extract pattern and choices.
	    // Keep any Variables in the pattern.
	    LList fullPattern = item.getPattern();
	    MatchEnv e = SimpleMatcher.match(activityPattern, fullPattern);
	    Debug.expect(e != null, "activity pattern doesn't match");
	    Object pattern = e.get(Symbol.intern("?pattern"));
	    List choices = (List)e.get(Symbol.intern("?values"));
	    Debug.expect(pattern != null, "lost pattern");
	    Debug.expect(choices != null, "lost choices");

	    // Find matches
	    List matches = new LinkedList();
	    for (Iterator i = choices.iterator(); i.hasNext();) {
		Object choice = i.next();
		MatchEnv ce = Matcher.match(pattern, choice);
		if (ce != null && !matches.contains(ce))
		    matches.add(ce);
	    }
	    Debug.expect(!matches.isEmpty());
	    Debug.noteln("Matches", matches);

	    // Tell the variable manager
	    Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	    MatchChoice mc = mm.addMatchChoice(matches);

	    item.setStatus(Status.COMPLETE);
	}

    }

}
