/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Sep 26 19:15:35 2008 by Jeff Dalton
 * Copyright: (c) 2008, AIAI, University of Edinburgh
 */

package ix.test;

import java.lang.reflect.Modifier;
import java.util.*;

import ix.ip2.Ip2XMLConfig;

import ix.icore.domain.Constraint;
import ix.icore.domain.TimeWindow;

import ix.iface.domain.LTF_Parser;

import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;
import ix.util.reflect.*;

/**
 * Generate a random instance of a class.
 */
public class MakeRandomInstance {

    protected XMLConfig config = new Ip2XMLConfig();

    protected ClassSyntax syntax = config.defaultClassSyntax();

    protected List<Class> relevantClasses =
        config.xmlSyntaxClasses(syntax, Object.class);

    protected InheritanceTree inheritance =
        new InheritanceTree(relevantClasses);

    protected LTF_Parser ltfParser = new LTF_Parser();

    protected long randomSeed = -4775043618027098404L;

    protected Random random = new Random(randomSeed);

    protected String LOWER = "abcdefghijklmnopqrstuvwxyz";

    protected String UPPER = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

    protected String DIGITS = "0123456789";

    protected String ALPHABET =
	LOWER + LOWER + UPPER + DIGITS +
	"     " +		// 5 spaces
	".,;:_?!";

    protected List<Class> randomObjectClasses =	Arrays.asList(new Class[] {
	    String.class, Long.class, Symbol.class, ItemVar.class
	});

    protected MinMax collectionSize = new MinMax(0, 2); // inclusive

    public MakeRandomInstance() {
    }

    public MakeRandomInstance randomize() {
	random.setSeed(System.currentTimeMillis());
	return this;
    }

    public static void main(String[] argv) {

	Debug.off();

        Parameters.processCommandLineArguments(argv);

	MakeRandomInstance maker = new MakeRandomInstance();
	StructuralEquality eq = new StructuralEquality();

        String xmltClassName = 
            Parameters.getParameter("xml-translator-class",
                                    "ix.util.xml.XMLTranslator");
        Class xmltClass =
            maker.config.defaultClassFinder().classForName(xmltClassName);

        XMLTranslator xmlt =
            (XMLTranslator)Util.makeInstance(xmltClass, maker.syntax);

        if (Parameters.haveParameter("collection-size"))
            maker.collectionSize =
                new MinMax(Parameters.getParameter("collection-size"));

	if (Parameters.getBoolean("randomize", false))
	    maker.randomize();

	String className = Parameters.getParameter("class");

	do {
	    String in = className != null ? className
		:  Util.askLine("Class name:");
	    if (in.equals("bye"))
		return;
	    Class c = XML.config().defaultClassFinder().classForName(in);
	    Object instance = maker.makeInstance(c);
	    String xml = xmlt.objectToXMLString(instance);
	    System.out.println(xml);
	    Object copy = xmlt.objectFromXML(xml);
	    Debug.expect(eq.equal(instance, copy), "instance and copy differ");
	} while (className == null);

    }

    public <T> T makeInstance(Class<T> c) {
	return (T)_makeInstance(c);
    }

    private Object _makeInstance(Class c) {
	if (c == Object.class)
	    return makeObject();
	ClassDescr cd = syntax.getClassDescr(c);
	if (cd.isPrimitive())
	    return makePrimitive(c);
	else if (cd.isCollection())
	    return makeCollection(c);
	else if (cd.isMap())
	    return makeMap(c);
	else if (cd.isAbstract()) // LList is abstract, so not testing earlier
            return makeConcreteInstance(c);
	else if (c == Constraint.class) // special case /\/
	    return makeConstraint();
	else if (c == TimeWindow.class) // special case /\/
	    return makeTimeWindow();
	else if (cd.isStruct())
	    return makeStruct(c);
	else if (cd.isInterface())
	    throw new IllegalArgumentException
		("Instance class is interface: " + c);
	else
	    return null;
    }

    protected Object makeConcreteInstance(Class c) {
        Debug.expect(Modifier.isAbstract(c.getModifiers()));
        List<Class> subclasses = inheritance.getSubclasses(c);
        if (subclasses == null)
            throw new IllegalArgumentException
		("Instance class is abstract: " + c);
        return makeInstance(randomth(subclasses));
    }

    protected <E> E randomth(List<E> items) {
        return items.get(random.nextInt(items.size()));
    }

    protected int randomIn(MinMax range) {
	return randomIn((int)range.getMin(), (int)range.getMax());
    }

    protected int randomIn(int min, int max) {
        return min + random.nextInt((max - min) + 1);
    }

    protected Object makeObject() {
	return makeInstance(randomth(randomObjectClasses));
    }

    protected Object makePrimitive(Class c) {
	if (c == String.class)
	    return makeMediumString();
	else if (SemiPrimitive.class.isAssignableFrom(c))
	    return makeSemiPrimitive(c);
	c = Fn.objectClass(c);
	if (c == Boolean.class)
	    return random.nextBoolean();
	else if (c == Integer.class)
	    return random.nextInt();
	else if (c == Long.class)
	    return random.nextLong();
	else if (c == Float.class)
	    return random.nextFloat();
	else if (c == Double.class)
	    return random.nextDouble();
// 	else if (c == Character.class)
// 	    ;
// 	else if (c == Short.class)
// 	    ;
// 	else if (c == Byte.class)
// 	    ;
	else
	    return null;
    }

    protected Object makeSemiPrimitive(Class c) {
	if (c == Symbol.class)
	    return Symbol.intern(makeShortString());
	else if (c == ItemVar.class)
	    return makeItemVar();
	else if (c == Name.class)
	    return makeName();
	else if (EnumeratedValue.class.isAssignableFrom(c)) {
	    List values = (List)Fn.applyStatic(c, "values", new Object[]{});
	    return randomth(values);
	}
	else if (c == Duration.class)
	    return makeDuration();
	else
	    throw new UnsupportedOperationException
		("Can't instantiate semiprimitive " + c);
    }

    public Name makeName() {
	return Name.valueOf(makeString(LOWER, randomIn(2, 6)));
    }

    public ItemVar makeItemVar() {
	return (ItemVar)Symbol.intern("?" + makeString(LOWER, 2));
    }

    public Duration makeDuration() {
	return new Duration(randomIn(0, 2 * 24 * 60 * 60 * 1000));
    }

    public String makeShortString() {
	return makeString(ALPHABET, random.nextInt(5));
    }

    public String makeMediumString() {
	return makeString(ALPHABET, 5 + random.nextInt(10));
    }

    public String makeString(String alphabet, int len) {
	StringBuilder b = new StringBuilder(len);
	int range = alphabet.length();
	for (int i = 0; i < len; i++)
	    b.append(alphabet.charAt(random.nextInt(range)));
	return b.toString();
    }

    public Collection makeCollection(Class c) {
	ClassDescr cd = syntax.getClassDescr(c);
	Class eltClass = cd.getEltType() != null
	    ? cd.getEltType().getDescribedClass()
	    : Object.class;
	return makeCollection(c, eltClass);
    }

    public Collection makeCollection(Class c, Class eltClass) {
	Class implClass = determineImplClass(c);
	if (LList.class.isAssignableFrom(implClass))
	    return makeLList(c, eltClass);
	Collection coll = (Collection)Util.makeInstance(implClass);
	for (int i = randomIn(collectionSize); i > 0; i--)
	    coll.add(makeInstance(eltClass));
	return coll;
    }

    protected Class determineImplClass(Class c) {
	if (c.isInterface()) {
	    if (TypedList.class.isAssignableFrom(c)) {
		Class preferred =
		    ListOf.findImplementationClass(c, LinkedList.class);
		if (preferred == null)
		    throw new UnsupportedOperationException
			("Cannot find a linked implementation for " + c);
		else
		    return preferred;
	    }
	    else if (c == List.class)
		return LinkedList.class;
	    else if (c == Set.class)
		return LinkedHashSet.class;
            else if (c == Map.class)
                return LinkedHashMap.class;
	    else
		throw new UnsupportedOperationException
		    ("Don't know how to implement " + c);
	}
	else
	    return c;
    }

    public LList makeLList(Class c, Class eltClass) {
	if (c == Null.class)
	    return Lisp.NIL;
	else if (c == LList.class || c == Cons.class) {
	    LListCollector coll =
		(LListCollector)makeCollection(LListCollector.class, eltClass);
	    return coll.contents();
	}
	else
	    throw new UnsupportedOperationException
		("Cannot handle LList class " + c);
    }

    public Map makeMap(Class c) {
        Class implClass = determineImplClass(c);
        Map map = (Map)Util.makeInstance(implClass);
	for (int i = randomIn(collectionSize); i > 0; i--)
            map.put(makeObject(),
                    makeObject());
	return map;
    }

    protected Object makeStruct(Class c) {
	ClassDescr cd = syntax.getClassDescr(c);
	Object struct = Util.makeInstance(c);
	for (Iterator i = cd.getFieldDescrs().iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    Object val = makeFieldValue(fd);
	    try {
		fd.setValue(struct, val);
	    }
	    catch (Exception e) {
		throw new RethrownException
		    ("Can't set " + fd + " in " + c, e);
	    }
	}
	return struct;
    }

    protected Object makeFieldValue(FieldDescr fd) {
	ClassDescr cd = fd.getTypeDescr();
	if (cd.isCollection() && cd.getEltType() != null)
	    return makeCollection(cd.getDescribedClass(),
				  cd.getEltType().getDescribedClass());
	else
	    return makeInstance(cd.getDescribedClass());
    }

    protected Constraint makeConstraint() {
	List parsers = ltfParser.getConstraintParsers().getCases();
	LTF_Parser.ConstraintParser chosenParser =
	    (LTF_Parser.ConstraintParser)randomth(parsers);
	Constraint template = chosenParser.makeTemplate();
	Constraint result = new Constraint
	    (template.getType(),
	     template.getRelation(),
	     makeConstraintParameters(template.getParameters()));
	result.setAnnotations(template.getAnnotations());
	return result;
    }

    protected List makeConstraintParameters(List parameterTemplates) {
	List parameters = new LinkedList();
	for (Object pt: parameterTemplates) {
	    Class parameterClass = pt instanceof Class
		? (Class)pt
		: pt.getClass();
	    parameters.add(makeInstance(parameterClass));
	}
	return parameters;
    }

    public TimeWindow makeTimeWindow() {
	switch (randomIn(0, 3)) {
	case 0: return new TimeWindow(makeItemVar(), makeDuration());
	case 1: return new TimeWindow(makeDuration(), makeItemVar());
	case 2: return new TimeWindow(makeItemVar(), makeItemVar());
	case 3:
	    Duration min = makeDuration();
	    long min_ms = min.asMilliseconds();
	    long max_ms = min_ms + randomIn(0, (int)min_ms);
	    return new TimeWindow(min, new Duration(max_ms));
	default:
	    throw new ConsistencyException("case out of range");
	}
    }

}
