/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Apr 25 21:47:56 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, 2006 AIAI, University of Edinburgh
 */

package ix.ispace;

import java.awt.Color;
import java.util.*;

import ix.ispace.event.*;
import ix.util.*;

/**
 * Keeps track of known relationships between agents
 * and agent capabilities.
 */
public class ContactManager {

    protected List listeners = new LinkedList();

    protected List agentData = new LinkedList();

    protected MultiMap relMap = new MultiHashMap();

    protected List groups = new LinkedList(); // remembers order added

    protected Map nameToGroup = new HashMap();

    protected ColorGenerator colorGen = new ColorGenerator();

    public ContactManager() {
	addStandardAgentGroups();
    }

    /**
     * Adds the agent as a contact.
     *
     * @throws AssertionFailure if the agent is already known.
     *
     * @see #addAgent(String, AgentRelationship)
     */
    public void addAgent(String name) {
	addAgent(name, AgentRelationship.CONTACT);
    }

    /**
     * Adds the agent, recording it as having the specified
     * relationship to this agent.
     *
     * @throws AssertionFailure if the agent is already known.
     */
    public void addAgent(String name, AgentRelationship rel) {
	Debug.expect(getAgentData(name) == null, "adding agent twice", name);
	AgentData data = new AgentData(name, rel);
	agentData.add(data);
	relMap.addValue(rel, data);
	fireNewContact(data);
    }

    /**
     * Adds the agents by calling {@link #addAgent(String, AgentRelationship)}.
     *
     * @throws AssertionFailure if any of the agents is already known.
     *
     * @see #noteAgent(String, AgentRelationship)
     * @see #ensureAgent(String, AgentRelationship)
     */
    public void addAgents(List names, AgentRelationship rel) {
	for (Iterator i = names.iterator(); i.hasNext();) {
	    addAgent((String)i.next(), rel);
	}
    }

    /**
     * Record the agent as a contact if it is not already known.
     * This makes it easier to reply to agents that have sent things
     * to us, e.g. by putting their name in menus of agents that
     * might be sent to.
     *
     * @see #noteAgent(String, AgentRelationship)
     */
    public void noteAgent(String name) {
	noteAgent(name, AgentRelationship.CONTACT);
    }

    /**
     * Record the agent as having the specified relationship to
     * this agent, if the agent is not already known.  The relationship
     * parameter is only an assumption or default, so if the agent
     * is already known, this method will not change its relationship.
     *
     * @see #addAgent(String, AgentRelationship)
     * @see #ensureAgent(String, AgentRelationship)
     */
    public void noteAgent(String name, AgentRelationship assumedRel) {
	AgentData data = getAgentData(name);
	if (data == null)
	    addAgent(name, assumedRel);
    }

    /**
     * Ensures that the agent is recorded as having the specified
     * relationship to this agent.  The agent is added is it is
     * not already known; otherwise the agent's relationship is
     * changed.
     *
     * @return true if the agent was not already known and hence
     *    had to be added.
     *
     * @see #noteAgent(String, AgentRelationship)
     * @see #addAgent(String, AgentRelationship)
     * @see #changeRelationship(String, AgentRelationship)
     */
    public boolean ensureAgent(String name, AgentRelationship rel) {
	AgentData known = getAgentData(name);
	if (known == null)
	    addAgent(name, rel);
	else
	    changeRelationship(name, rel);
	return known == null;
    }

    public List getAgentData() {
	return agentData;
    }

    public AgentData getAgentData(String name) {
	for (Iterator i = agentData.iterator(); i.hasNext();) {
	    AgentData data = (AgentData)i.next();
	    if (data.getName().equals(name))
		return data;
	}
	return null;
    }

    public List getAgentData(AgentRelationship rel) {
	return (List)relMap.get(rel);
    }

    public List getAgentData(Capability c) {
	return getAgentData(c, false);
    }

    public List getAgentData(Capability c, boolean defaultIfUnknown) {
	List result = new LinkedList();
	for (Iterator i = agentData.iterator(); i.hasNext();) {
	    AgentData data = (AgentData)i.next();
	    if (data.hasCapability(c, defaultIfUnknown))
		result.add(data);
	}
	return result;
    }

    public List getSortedNameList() {
	List names = (List)Collect
	    .map(agentData, Fn.accessor(AgentData.class, "getName"));
	Collections.sort(names);
	return names;
    }

    protected void replaceAgentData(AgentData oldData, AgentData newData) {
	Collect.replaceFirst(oldData, newData, agentData);
	relMap.removeValue(oldData.getRelationship(), oldData);
	relMap.addValue(newData.getRelationship(), newData);
    }

    public void changeRelationship(String name, AgentRelationship newRel) {
	AgentData oldData = getAgentData(name);
	AgentRelationship oldRel = oldData.getRelationship();

	Debug.noteln("Changing " + name + " from " + oldRel + " to " + newRel);

	if (newRel != oldRel) {
	    AgentData newData = new AgentData(oldData);
	    newData.setRelationship(newRel);
	    replaceAgentData(oldData, newData);
	    checkAgentData(name, newData);
	    fireContactChange(oldData, newData);
	}
    }

    public void deleteAgent(String name) {
	AgentData oldData = getAgentData(name);
	if (oldData == null)
	    throw new IllegalArgumentException
		("Attempt to delete unknown agent " + name);
	AgentRelationship oldRel = oldData.getRelationship();

	agentData.remove(oldData);
	relMap.removeValue(oldRel, oldData);

	fireContactDeleted(oldData);
    }

    public void setCapabilities(String name, List newCapabilities) {
	AgentData oldData = getAgentData(name);
	List oldCapabilities = oldData.getCapabilities();

	Debug.noteln("Changing " + name + " capabilities from "
		     + oldCapabilities + " to " + newCapabilities);

	if (!Collect.equalAsSets(oldCapabilities, newCapabilities)) {
	    // Capabilities will change
	    AgentData newData = new AgentData(oldData);
	    newData.setCapabilities(newCapabilities);
	    replaceAgentData(oldData, newData);
	    checkAgentData(name, newData);
	    fireContactChange(oldData, newData);
	}
    }

    private void checkAgentData(String name, AgentData newData) {
	// Check that new data is found
	Debug.expect(getAgentData(name) == newData,
		     "Don't get new agent data for", name);
	// Check capaibilities
	List newCapabilities = newData.getCapabilities();
	Debug.noteln("Checking that " + name + " has capabilities " +
		     newCapabilities);
	for (Iterator i = newCapabilities.iterator(); i.hasNext();) {
	    Capability c = (Capability)i.next();
	    Debug.expect(getAgentData(c).contains(newData),
			 "Agent lacks capability", c);
	}
	// Check relationship
	// This one was once a bug:
	Debug.expect(getAgentData(newData.getRelationship())
		     .contains(newData),
	             "Relationship doesn't give new data");
    }

    /*
     * Colors -- used for message text, for example.
     */

    // /\/: For now, this is separate from the agent-data.

    public Color getAgentColor(String name) {
	return colorGen.getColor(name);
    }

    public Color getAgentColor(String name, Color defaultColor) {
	return colorGen.getColor(name, defaultColor);
    }

    public void setAgentColor(String name, Color color) {
	colorGen.setColor(name, color);
    }


    /*
     * AgentGroups
     */

    public Collection getAgentGroups() {
	// return nameToGroup.values();
	return groups;
    }

    public AgentGroup getAgentGroup(String name) {
	return (AgentGroup)nameToGroup.get(name);
    }

    public void addAgentGroup(AgentGroup group) {
	groups.add(group);
	nameToGroup.put(group.getName(), group);
    }

    protected void addStandardAgentGroups() {

	// All known agents
	addAgentGroup(new AbstractAgentGroup("all", this) {});

	// A group for each relationship
	for (Iterator i = AgentRelationship.values().iterator();
	     i.hasNext();) {
	    AgentRelationship rel = (AgentRelationship)i.next();
	    if (rel == AgentRelationship.NONE) // /\/
		continue;
	    addAgentGroup(new RelationshipGroup(rel, this));
	}

    }


    /**
     * Process command-line arguments that are about our relationships
     * with other agents.
     *
     * <p>For each AgentRelationship, R, R+"s" is treated as a parameter
     * whose value should be a comma-separated list of names.
     *
     * <p>Simple capabilities can be given by
     * <pre>
     *    -external-capabilities=<i>ipcName</i>:<i>verb</i>, ...
     * </pre>
     *
     * @see ix.ispace.AgentRelationship
     */
    public void processCommandLineArguments() {
	// AgentRelationships
	for (Iterator i = AgentRelationship.values().iterator();
	     i.hasNext();) {
	    AgentRelationship rel = (AgentRelationship)i.next();
	    // /\/: Assumes +"s" plurals
	    List names = Parameters.getList(rel + "s");
	    warnAboutQuestionableAgentNames(names, rel);
	    addAgents(names, rel);
	}
	// external-capabilities=ipcName:verb,ipcName:verb,...
	String caps = Parameters.getParameter("external-capabilities", "");
	for (Iterator i = Strings.breakAt(",", caps).iterator();
	     i.hasNext();) {
	    String cap = (String)i.next();
	    String[] parts = Strings.breakAtFirst(":", cap);
	    String ipcName = parts[0];
	    String verb = parts[1];
	    noteAgent(ipcName, AgentRelationship.SERVICE);
	    AgentData data = getAgentData(ipcName);
	    data.addCapability(new VerbCapability(verb));
	}
    }

    protected void warnAboutQuestionableAgentNames(List names,
						   AgentRelationship rel) {
	List questionable = new LinkedList();
	for (Iterator i = names.iterator(); i.hasNext();) {
	    String name = (String)i.next();
	    if (name.indexOf(':') >= 0)
		questionable.add(name);
	}
	if (!questionable.isEmpty())
	    Debug.warn("The following agent names, gives as " + rel + "s, " +
		       "contain colons: " +
		       Strings.conjunction(questionable) + ".  " +
		       "Perhaps you meant them as agent capabilities.");
    }

    public void addContactListener(ContactListener listener) {

	listeners.add(listener);
    }

    public void fireNewContact(AgentData data) {
	fireContactEvent(new ContactEvent(this, null, data));
    }

    public void fireContactDeleted(AgentData data) {
	fireContactEvent(new ContactEvent(this, data, null));
    }

    public void fireContactChange(AgentData oldData, AgentData newData) {
	fireContactEvent(new ContactEvent(this, oldData, newData));
    }

    public void fireContactEvent(ContactEvent event) {
	Debug.noteln("fireContactEvent", event);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    ContactListener listener = (ContactListener)i.next();
	    listener.contactChange(event);
	}
    }

}
