/* Author: Gerhard Wickler
 * I-Sim Adaptation: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Mar 29 00:19:31 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim.dynamic;

import java.awt.event.*;
import java.io.*;
import java.util.*;

import javax.swing.*;

import ix.isim.*;

import ix.icore.domain.*;

import ix.iface.util.IFUtil;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * <p>I-X Simple Object Animation Tool</p>
 *
 * @author Gerhard Wickler
 * @version 1.0
 */
public class ObjectAnimator implements SimulationEngine {

    // the List of AmimatedObjects
    private List allAnimated = new LinkedList();

    private static ISim isim;	// static but shouldn't be /\/

    private JMenuItem startStop;

    public ObjectAnimator(ISim isim) {
	this.isim = isim;
    }

    public void init() {

        // add the menu entrys for animated objects:
	JFrame frame = isim.getFrame();
	JMenuBar bar = frame.getJMenuBar();
	JMenu menu = new JMenu("Sim");
	bar.add(menu, 2);
	ActionInterpreter act = new ActionInterpreter();
	menu.add(IFUtil.makeMenuItem("Load Dynamic Objects ...", act));
	menu.add(startStop = 
		 IFUtil.makeMenuItem("Start Dynamic Objects", act));

        // load objects if such parameter was given:
        if (Parameters.haveParameter("dynamic-objects")) {
            String objFile = Parameters.getParameter("dynamic-objects");
            try {
                allAnimated.addAll(loadObjectFile(new FileReader(objFile)));
            }
	    catch (FileNotFoundException e) {
                Debug.displayException(e);
            }
        }

    }

    public void startSimulation() {
        ListIterator li = allAnimated.listIterator();
        while (li.hasNext()) {
            ((AnimatedObject) li.next()).start();
        }
    }

    public void stopSimulation() {
	for (Iterator aoi = allAnimated.iterator(); aoi.hasNext(); ) {
	    ((AnimatedObject) aoi.next()).terminate();
	}
	allAnimated.clear();
    }

    class ActionInterpreter implements ActionListener {

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("ObjectAnimator action:", command);
	    if (command.startsWith("Start")) {
		startStop.setText("Stop Dynamic Objects");
		startSimulation();
	    }
	    else if (command.startsWith("Stop")) {
		stopSimulation();
	    }
	    else if (command.startsWith("Load")) {
		loadObjects();
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for", command);
	}

    }

    public static synchronized void sendConstraint(
            String att, String sName, double val, String target) {
	// /\/: We don't actually send it, and, because this is static,
	// the isim field has to be static.
	Constraint c = makeConstraint(att, sName, val);
	try {
	    synchronized(isim) {
		isim.handleNewConstraint(c);
	    }
	}
	catch (Throwable t) {
	    Debug.displayException(t);
	}
    }

    public static Constraint makeConstraint(String att, String sName,
					    double val) {
	LList pattern = Lisp.list(Symbol.intern(att), Symbol.intern(sName));
        PatternAssignment pa = new PatternAssignment(pattern, new Double(val));
	return new Constraint("world-state", "effect", Lisp.list(pa));
    }

    protected void loadObjects() {
	XMLLoader loader = new XMLLoader(isim.getFrame(),
					 ISim.AnimatedObjectList.class);
	List objects = (List)loader.loadObject();
	if (objects != null)
	    allAnimated.addAll(objects);
    }

    public static ISim.AnimatedObjectList loadObjectFile(Reader r) {
        ISim.AnimatedObjectList result = new ISim.AnimatedObjectList();
        StreamTokenizer st = new StreamTokenizer(r);
        try {
            st.nextToken();
            while ((st.ttype == StreamTokenizer.TT_WORD) &&
                   (st.sval.equals("animate"))) {
                st.nextToken();
                result.add(new AnimatedObject(st));
            }
            if (st.ttype != StreamTokenizer.TT_EOF) {
                throw new java.text.ParseException(
                        "EOF expected", st.lineno());
            }
        } catch (IOException ioe) {
            Debug.displayException
		("Unable to read objects from file because", ioe);
        } catch (java.text.ParseException pe) {
            Debug.displayException
		("Syntax error in object animation file", pe);
        }
        Debug.noteln("Nr of animated objects read: " + result.size());
        return result;
    }

}
