/* Author: Stephen Potter
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import java.util.LinkedList;
import java.util.ListIterator;

import ix.icore.Activity;
import ix.icore.IXAgent;
import ix.util.Debug;
import ix.util.IPC;
import ix.util.lisp.*;

public class ISimTimerServer extends ISimTimer {

    private LinkedList registryList;

    protected ISimTimerServer(IXAgent agent) {
        super(agent);
        initializeRegistry();
    }


    // overwriting ISimTimer methods:
    public void start(long simTime, double factor) throws ISimTimerException {
        super.start(simTime, factor);
        sendStartActivity(simTime, factor);
    }

    public long pause() {
        long result = super.pause();
        sendPauseActivity(result);
        return result;
    }

    public long resume() {
        long result = super.resume();
        sendResumeActivity(result);
        return result;
    }

    public long changeAccelerationFactor(double acceleration) {
        long result = super.changeAccelerationFactor(acceleration);
        sendAccelerateActivity(acceleration);
        return result;
    }

    public void terminateTimer() {
        super.terminateTimer();
        sendStopActivity();
    }


    private void initializeRegistry() {
        registryList = new LinkedList();
    }

    // what to set up data structure for storing registered agents,
    // plus some sort of listener method for handling incoming register
    // activities.
    // Assumes registration activities are of the form:
    // synchronize-simulation-time register <agent-symbol-name>

    public void handleSynchronizeActivity(Activity synchActivity) {
        Symbol verb = (Symbol) synchActivity.getVerb();
        Debug.expectSame(SYNCHRONIZE_LABEL, verb);

        LList parameters = synchActivity.getParameters();
        // parameters should be "register", <agent-name>:
        String param1 = (String) parameters.get(0);

        if (!param1.equalsIgnoreCase(REGISTER)) {
            Debug.noteln("Unrecognised synchronization activity: " + param1 +
                         ". Ignoring.");
        } else {
            String param2 = (String) parameters.get(1);
            registerAgent(param2);
        }

    }


    synchronized public void registerAgent(String agentName) {
        // assume that agentName is something that we can directly send
        // messages to (eg. a symbol-name for the XML strategy, or a
        // full JID for Jabber). (The I-X agent symbol name should be
        // adequate.)

        if (!registryList.contains(agentName)) {

            // And then simply (for now) add the agentName to the end of the list:
            registryList.add(agentName);
            Debug.noteln(agentName + " has been registered with I-Sim.");
        }

        // First, we want to check if the server timer has already started;
        // if so, we want to send a "start" message to the new agent with
        // the current simulation time and acceleration factor:

        if (simulationStarted()) {
            long simTime = getSimTimeInMillis();
            Activity startActivity = new Activity(Lisp.list(
                    SYNCHRONIZE_LABEL, START,
                    Long.toString(simTime),
                    Double.toString(getAccelerationFactor())));

            IPC.sendObject(agentName, startActivity);

            if (!simulationActive) {
                Activity pauseActivity =
                        new Activity(Lisp.list(SYNCHRONIZE_LABEL, PAUSE,
                                               Long.toString(simTime)));
                IPC.sendObject(agentName, pauseActivity);
            }
        }

    }


    // want methods that send the different types of synchronize activities
    // to each of the registered agents.
    // The following are the recognised activities:
    // synchronize-simulation-time start <sim-time> <accel-factor>
    //                             pause <sim-time>
    //                             resume <sim-time>
    //                             accelerate <accel-factor>
    //                             stop [? - not sure we need this]


    public void sendStartActivity(long simTime, double factor) {
        Activity startActivity =
                new Activity(Lisp.list(SYNCHRONIZE_LABEL, START,
                                       Long.toString(simTime),
                                       Double.toString(factor)));
        iterateSendMessage(startActivity);
    }

    public void sendPauseActivity(long simTime) {
        Activity pauseActivity =
                new Activity(Lisp.list(SYNCHRONIZE_LABEL, PAUSE,
                                       Long.toString(simTime)));
        iterateSendMessage(pauseActivity);
    }

    public void sendResumeActivity(long simTime) {
        Activity resumeActivity
                = new Activity(Lisp.list(SYNCHRONIZE_LABEL, RESUME,
                                         Long.toString(simTime)));
        iterateSendMessage(resumeActivity);
    }

    public void sendAccelerateActivity(double factor) {
        Activity accelerateActivity =
                new Activity(Lisp.list(SYNCHRONIZE_LABEL, ACCELERATE,
                                       Double.toString(factor)));
        iterateSendMessage(accelerateActivity);
    }

    public void sendStopActivity() {
        Activity stopActivity =
                new Activity(Lisp.list(SYNCHRONIZE_LABEL, STOP));
        iterateSendMessage(stopActivity);
    }


    synchronized private void iterateSendMessage(Activity msg) {
        // iterates through the current registryList and sends message
        // msg to each.

        ListIterator liter = registryList.listIterator();

        while (liter.hasNext()) {
            String agentname = (String) liter.next();
            try {
                IPC.sendObject(agentname, msg);
            } catch (IPC.IPCException ipce) {
                liter.remove();
            }
        }
    }
}
