package ix.isim;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.text.DateFormat;
import java.util.*;

import javax.swing.*;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumnModel;

import ix.icore.Activity;
import ix.icore.IXAgent;
import ix.iface.util.ToolFrame;
import ix.isim.util.LongPriorityQueue;
import ix.util.Debug;
import ix.util.Duration;

public class ISimEventsFrame extends ToolFrame {

    /** the file chooser used for selecting an events-file **/
    static protected JFileChooser fileSelect = new JFileChooser();
    static {
        fileSelect.setDialogTitle("Select events files");
        fileSelect.setCurrentDirectory(new File("./scenarios"));
        fileSelect.setFileFilter(new ix.util.FileExtensionFileFilter("lsp"));
        fileSelect.setAcceptAllFileFilterUsed(false);
        fileSelect.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fileSelect.setMultiSelectionEnabled(true);
    }

    protected ISimEngine simulator;

    JMenuBar mainMenu = new JMenuBar();
    JMenu fileMenu = new JMenu();
    JMenuItem loadMenuItem = new JMenuItem();
    JMenuItem closeMenuItem = new JMenuItem();
    JMenu simulationMenu = new JMenu();
    JMenuItem startMenuItem = new JMenuItem();
    JMenuItem pauseMenuItem = new JMenuItem();
    JMenuItem resumeMenuItem = new JMenuItem();
    JMenu eventsMenu = new JMenu();
    JMenuItem clearMenuItem = new JMenuItem();
    JMenuItem nextMenuItem = new JMenuItem();

    BorderLayout UILayout = new BorderLayout();
    JScrollPane tableScrollPane = new JScrollPane();

    JTable eventsTable = new JTable();
    EventsTableModel eventsTableModel;

    public ISimEventsFrame(ISimEngine simulator) {
        this.simulator = simulator;
        this.eventsTableModel = new EventsTableModel(simulator);
        try {
            jbInit();
        } catch (Exception exception) {
            exception.printStackTrace();
        }
    }

    private void jbInit() throws Exception {
        this.setTitle(IXAgent.getAgent().getAgentDisplayName() +
                      " I-Sim Control");
        this.setSize(new Dimension(750, 600));

        loadMenuItem.setText("Load Timed Events ...");
        loadMenuItem.addActionListener(
                new ISimEventsFrame_loadMenuItem_actionAdapter(this));
        closeMenuItem.setText("Close");
        closeMenuItem.addActionListener(
                new ISimEventsFrame_closeMenuItem_actionAdapter(this));
        fileMenu.setText("File");
        fileMenu.add(loadMenuItem);
        fileMenu.addSeparator();
        fileMenu.add(closeMenuItem);
        startMenuItem.setText("Start Simulation ...");
        startMenuItem.addActionListener(
                new ISimEventsFrame_startMenuItem_actionAdapter(this));
        pauseMenuItem.setText("Pause Simulation");
        pauseMenuItem.setEnabled(false);
        pauseMenuItem.addActionListener(
                new ISimEventsFrame_pauseMenuItem_actionAdapter(this));
        resumeMenuItem.setText("Resume Simulation");
        resumeMenuItem.setEnabled(false);
        resumeMenuItem.addActionListener(
                new ISimEventsFrame_resumeMenuItem_actionAdapter(this));
        simulationMenu.setText("Simulation");
        simulationMenu.add(startMenuItem);
        simulationMenu.add(pauseMenuItem);
        simulationMenu.add(resumeMenuItem);
        clearMenuItem.setText("Clear All Waiting");
        clearMenuItem.addActionListener(
                new ISimEventsFrame_clearMenuItem_actionAdapter(this));
        nextMenuItem.setText("Execute Next Now");
        nextMenuItem.addActionListener(
                new ISimEventsFrame_nextMenuItem_actionAdapter(this));
        eventsMenu.setText("Events");
        eventsMenu.add(clearMenuItem);
        eventsMenu.add(nextMenuItem);
        mainMenu.add(fileMenu);
        mainMenu.add(simulationMenu);
        mainMenu.add(eventsMenu);
        setJMenuBar(mainMenu);

        getContentPane().setLayout(UILayout);

        eventsTable.setModel(eventsTableModel);
        eventsTable.setAutoCreateColumnsFromModel(true);
        eventsTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        eventsTable.setRowHeight(16);
        eventsTable.setCellSelectionEnabled(false);
        eventsTable.setColumnSelectionAllowed(false);
        eventsTable.setRowSelectionAllowed(true);

        TableColumnModel tcm = eventsTable.getColumnModel();
        tcm.getColumn(0).setMinWidth(100);
        tcm.getColumn(0).setMaxWidth(100);
        tcm.getColumn(1).setMinWidth(100);
        tcm.getColumn(1).setMaxWidth(100);
        tcm.getColumn(2).setMinWidth(100);
        tcm.getColumn(3).setMinWidth(750);

        tableScrollPane.getViewport().setBackground(Color.white);
        tableScrollPane.getViewport().add(eventsTable);

        getContentPane().add(tableScrollPane, java.awt.BorderLayout.CENTER);
    }

    public void loadMenuItem_actionPerformed(ActionEvent e) {
        Debug.noteln("ISim action:", e.getActionCommand());
        if (fileSelect.showDialog(this, "Load") == JFileChooser.APPROVE_OPTION) {
            simulator.loadTimedEvents(fileSelect.getSelectedFiles());
            Thread.currentThread().yield();
        }
    }

    public void closeMenuItem_actionPerformed(ActionEvent e) {
        setVisible(false);
    }

    public void startMenuItem_actionPerformed(ActionEvent e) {
        Debug.noteln("ISim action:", e.getActionCommand());
        // get the start time for the simulation:
        StartSimDialog startSim =
                new StartSimDialog(this, "Start Simulation", true);
        startSim.show();
        if (startSim.startSelected) {
            startMenuItem.setEnabled(false);
            // start the timer + simulation:
            simulator.startSimulation(
                    startSim.startTime, startSim.acceleration);
            Thread.currentThread().yield();
            // change the active menu to pause:
            pauseMenuItem.setEnabled(true);
        }
    }

    public void pauseMenuItem_actionPerformed(ActionEvent e) {
        Debug.noteln("ISim action:", e.getActionCommand());
        pauseMenuItem.setEnabled(false);
        // pause the simulation + timer:
        simulator.pauseSimulation();
        simulator.theISim.getISimTimer().pause();
        Thread.currentThread().yield();
        // change the active menu to resume:
        resumeMenuItem.setEnabled(true);
    }

    public void resumeMenuItem_actionPerformed(ActionEvent e) {
        Debug.noteln("ISim action:", e.getActionCommand());
        resumeMenuItem.setEnabled(false);
        // resume the simulation + timer:
        simulator.theISim.getISimTimer().resume();
        simulator.resumeSimulation();
        Thread.currentThread().yield();
        // change the active menu to pause:
        pauseMenuItem.setEnabled(true);
    }

    public void clearMenuItem_actionPerformed(ActionEvent e) {
        Debug.noteln("ISim action:", e.getActionCommand());
        simulator.clearAllWaitingEvents();
    }

    public void nextMenuItem_actionPerformed(ActionEvent e) {
        Debug.noteln("ISim action:", e.getActionCommand());
        simulator.executeNextEvent();
    }

    public void makeRowsFromQueue(
            java.util.List completedEvents,
            java.util.List activeEvents,
            LongPriorityQueue waitingEvents) {
        ((EventsTableModel) eventsTable.getModel()).makeRowsFromQueue(
                completedEvents, activeEvents, waitingEvents);
        Thread.currentThread().yield();
    }
}


class EventsTableModel extends AbstractTableModel {

    final static int NROFCOLS = 4;

    final static int STATUS = 0;
    final static int TIME = 1;
    final static int THREAD = 2;
    final static int EVENT = 3;

    final static String[] titles =
            new String[] {
            "Status", "Time", "Thread", "Event"};
    final static Class[] types =
            new Class[] {
            String.class, String.class, String.class, Activity.class};

    // used for creating the String representing simulated time:
    static private DateFormat df = DateFormat.getDateTimeInstance(
            DateFormat.SHORT, DateFormat.SHORT);

    // the TimedEvents in this table:
    protected java.util.List events = new ArrayList();
    // map evetns to the row they are in:
    protected java.util.Map eventRows = new HashMap();

    private ISimEngine simulator;

    public EventsTableModel(ISimEngine simulator) {
        super();
        this.simulator = simulator;
    }

    public int getColumnCount() {
        return NROFCOLS;
    }

    public String getColumnName(int col) {
        return titles[col];
    }

    public Class getColumnClass(int col) {
        return types[col];
    }

    public int getRowCount() {
        return events.size();
    }

    synchronized public void eventUpdated(TimedEvent event) {
        Integer row = (Integer) eventRows.get(event);
        if (row != null) {
            fireTableRowsUpdated(row.intValue(), row.intValue());
        } else {
            simulator.updateUI();
        }
        Thread.currentThread().yield();
    }

    public Object getValueAt(int row, int col) {
        TimedEvent event = (TimedEvent) events.get(row);
        switch (col) {
        case STATUS:
            return TimedEvent.statusString[event.getStatus()];
        case TIME:
            long startT = simulator.getSimStartTime();
            if (startT != 0L) {
                return df.format(new Date(startT + event.timeInMillis));
            }
            return (new Duration(event.timeInMillis)).toISOString();
        case THREAD:
            return event.getThread();
        case EVENT:
            return event.getActivity();
        }
        throw new IllegalArgumentException("Column not in model.");
    }

    synchronized public void makeRowsFromQueue(
            java.util.List completedEvents,
            java.util.List activeEvents,
            LongPriorityQueue waitingEvents) {
        java.util.List newEvents = new ArrayList();
        java.util.Map newEventRows = new HashMap();
        int index = 0;
        for (Iterator eventI = completedEvents.iterator(); eventI.hasNext(); ) {
            TimedEvent event = (TimedEvent) eventI.next();
            newEvents.add(event);
            newEventRows.put(event, new Integer(index++));
        }
        for (Iterator eventI = activeEvents.iterator(); eventI.hasNext(); ) {
            TimedEvent event = (TimedEvent) eventI.next();
            newEvents.add(event);
            newEventRows.put(event, new Integer(index++));
        }
        for (Iterator eventI = waitingEvents.elements(); eventI.hasNext(); ) {
            TimedEvent event = (TimedEvent) eventI.next();
            newEvents.add(event);
            newEventRows.put(event, new Integer(index++));
        }
        events = newEvents;
        eventRows = newEventRows;
        fireTableDataChanged();
        Thread.currentThread().yield();
    }
}


class ISimEventsFrame_loadMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_loadMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.loadMenuItem_actionPerformed(e);
    }
}


class ISimEventsFrame_closeMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_closeMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.closeMenuItem_actionPerformed(e);
    }
}


class ISimEventsFrame_startMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_startMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.startMenuItem_actionPerformed(e);
    }
}


class ISimEventsFrame_pauseMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_pauseMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.pauseMenuItem_actionPerformed(e);
    }
}


class ISimEventsFrame_resumeMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_resumeMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.resumeMenuItem_actionPerformed(e);
    }
}

class ISimEventsFrame_clearMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_clearMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.clearMenuItem_actionPerformed(e);
    }
}

class ISimEventsFrame_nextMenuItem_actionAdapter implements ActionListener {
    private ISimEventsFrame adaptee;
    ISimEventsFrame_nextMenuItem_actionAdapter(ISimEventsFrame adaptee) {
        this.adaptee = adaptee;
    }

    public void actionPerformed(ActionEvent e) {
        adaptee.nextMenuItem_actionPerformed(e);
    }
}
