/* Generic wrapper for simple information source I-X agents.
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Sep 12 15:15:55 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.iquery;

import java.util.*;

import ix.iface.util.Reporting;

import ix.icore.*;
import ix.icore.domain.Constraint;
import ix.icore.domain.PatternAssignment;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Performs a lookup for a query agent.
 *
 * <p>See {@link IQuery} for an example.
 */
public abstract class LookupHandler {

    /** The agent that created this handler. */
    protected IQuery queryAgent;

    /** The activity this handler is processing. */
    protected Activity lookupActivity;

    /**
     * Creates a LookupHander to handle the specified activity
     * for the specified agent.
     */
    public LookupHandler(IQuery queryAgent, Activity lookupActivity) {
	this.queryAgent = queryAgent;
	this.lookupActivity = lookupActivity;
    }

    /**
     * Adds a line to the agent's transcript window after switching
     * to the AWT event-handling thread.
     *
     * @see Util#swingAndWait(Runnable)
     */
    protected void transcript(final String line) {
	Util.swingAndWait(new Runnable() {
	    public void run() {
		do_transcript(line);
	    }
	});
    }

    private void do_transcript(String line) {
	queryAgent.do_transcript(line);
    }


    /* Report utilities */

    /**
     * Sends a report about this handler's {@link #lookupActivity}
     * by calling {@link #sendReport(Activity, ReportType, String)}.
     */
    protected void sendReport(ReportType type, String text) {
	sendReport(lookupActivity, type, text);
    }

    /**
     * Sends a report about the specified activity.
     *
     * @see #makeReport(Activity, ReportType, String)
     * @see #sendAbout(Activity, Object)
     */
    protected void sendReport(final Activity about,
			      final ReportType type,
			      final String text) {
	sendAbout(about, makeReport(about, type, text));
    }

    /**
     * Constructs a report about this handler's {@link #lookupActivity}.
     */
    protected Report makeReport(ReportType type, String text) {
	return makeReport(lookupActivity, type, text);
    }

    /**
     * Constructs a report about the specified activity.
     */
    protected Report makeReport(Activity about, ReportType type, String text) {
        Name ourName = Name.valueOf(queryAgent.getAgentIPCName());
	Report r = new Report(text);
	r.setReportType(type);
	r.setRef(about.getRef());
	r.setSenderId(ourName);
	return r;
    }


    /* Communication utilities */

    /**
     * Sends to the agent that send this handler's {@link #lookupActivity}.
     *
     * @see #sendAbout(Activity, Object)
     */
    protected void sendReply(Object contents) {
	sendAbout(lookupActivity, contents);
    }

    /**
     * Sends to the agent that should receive reports about the
     * specified activity.  The object being sent needn't be a
     * Report, but typically will be one.  Any fields in the
     * object that refer to the asctivity must have already
     * been filled-in; this method sends the object as-is.
     *
     * <p>This method calls {@link #sendTo(Object, Object)}.
     */
    protected void sendAbout(Activity about, Object contents) {
	sendTo(about.getSenderId(), contents);
    }

    /**
     * Sends the contents object to the agent represented by the
     * specified destination.  The object is sent by calling
     * {@link IPC#sendObject(Object, Object)} after switching
     * to the AWT event-handling thread.
     *
     * @see Util#swingAndWait(Runnable)
     */
    protected void sendTo(final Object destination, final Object contents) {
	// /\/: Not clear we really need to go to the event thread to do this,
	// apart from the transcript part.
	Util.swingAndWait(new Runnable() {
	    public void run() {
		do_transcript("Sending " + Reporting.description(contents));
		String toName = destination.toString(); // canonical
		IPC.sendObject(toName, contents);
	    }
	});
    }


    /* Constraint utilities */

    protected Constraint makeStateConstraint(LList pattern, Object value) {
	return new Constraint
	              ("world-state", "effect",
		       Lisp.list(new PatternAssignment(pattern, value)));
    }

    protected Constraint makeStateConstraint(LList pattern) {
	return new Constraint
	              ("world-state", "effect",
		       Lisp.list(new PatternAssignment(pattern)));
    }

    protected Constraint makeStateConstraint(String pattern, Object value) {
	return makeStateConstraint(Lisp.elementsFromString(pattern), value);
    }

    protected Constraint makeStateConstraint(String pattern) {
	return makeStateConstraint(Lisp.elementsFromString(pattern));
    }


    /* Here's what a subclass must define. */

    /**
     * Looks up information and sends report back to the agent
     * that sent us the activity.  Note that this method is
     * called in a separate thread and is a method of a new
     * instance of the relevant LookupHandler subclass.
     * This allows multiple activities to be handled in
     * parallel without interfering with each other.
     */
    protected abstract void handleLookup();

}
