/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Dec  3 04:46:09 2005 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.event.HyperlinkEvent;

import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.html.StyleSheet;

import javax.swing.text.StyleConstants;
import javax.swing.text.Style;

import java.awt.Container;
import java.awt.BorderLayout;

import java.awt.event.*;

import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.iplan.event.*;

import ix.iface.util.ToolFrame;
import ix.iface.util.IconImage;
import ix.iface.util.IFUtil;
import ix.iface.util.HtmlStringWriter;
import ix.iface.util.IXHtmlEditorKit;

import ix.util.*;

/**
 * A control and comparison tool for options.
 */
public class IPlanOptionTool {

    Ip2 ip2;
    IPlanOptionManager optMan;

    OptionToolFrame frame;
    OptionTree optionTree;
    OptionMatrix optionMatrix;

    public IPlanOptionTool(Ip2 ip2) {
	this.ip2 = ip2;

	optMan = ip2.getOptionManager();
	Debug.expect(optMan != null, "No option manager");

	optionTree = new OptionTree();
	optionMatrix = new OptionMatrix();
	frame = new OptionToolFrame();

	optMan.addOptionListener(optionTree);
	optMan.addOptionListener(optionMatrix);

	ip2.addResetHook(new ResetHook());

//  	frame.noteCurrentOption(ip2.getOptionManager().getOption());
    }

    public void setVisible(boolean v) {
	frame.setVisible(v);
    }

    protected class ResetHook implements Runnable {
	// Contexts have already been cleared by the time this is is called.
	public void run() {
	    Debug.noteln("Resetting", IPlanOptionTool.this);
	    frame.reset();
	}
    }

    class OptionToolFrame extends ToolFrame implements ActionListener {

	OptionUI optionUI;

	Container contentPane;

	OptionToolFrame() {
	    super(ip2.getAgentDisplayName() + " I-Plan Option Tool");
	
	    // /\/: optionUI needs some extra initialisation,
	    // because it isn't created until this tool is first
	    // used.
	    optionUI = new OptionUI(optMan, this);
	    optionUI.populateSelectOptionMenu();
	    optionUI.noticeCurrentOptionName
		(new OptionEvent(optMan, optMan.getOption()));

	    setIconImage(IconImage.getIconImage(this));
	    setJMenuBar(makeMenuBar());

	    contentPane = getContentPane();

	    JScrollPane treeScroll = new JScrollPane(optionTree);
	    treeScroll.setBorder
		(BorderFactory.createTitledBorder("Option Tree"));

	    JScrollPane matrixScroll = new JScrollPane(optionMatrix);
	    matrixScroll.setBorder
		(BorderFactory.createTitledBorder("Options"));

	    JSplitPane split = new JSplitPane
		(JSplitPane.HORIZONTAL_SPLIT, treeScroll, matrixScroll);
	    split.setOneTouchExpandable(true);
	    split.setResizeWeight(0.1);

	    contentPane.add(split, BorderLayout.CENTER);

	    pack();
	    setSize(500, 300);
	    validate();
	    split.setDividerLocation(0.33);
	}

	void reset() {
	    optionTree.reset();
	    optionMatrix.reset();
	}

	protected JMenuBar makeMenuBar() {
	    JMenuBar bar = new JMenuBar();
	    JMenu fileMenu = new JMenu("File");
	    bar.add(fileMenu);
	    fileMenu.add(IFUtil.makeMenuItem("Close", this));
	    bar.add(optionUI.getOptionMenu());
	    return bar;
	}

	/** Action interpreter */
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("I-Plan Option Tool frame action:", command);
	    if (command.equals("Close")) {
		frame.setVisible(false);
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

    }

    /*
     * Option tree
     */

    class OptionTree extends JTree implements TreeSelectionListener,
					      OptionListener {

	DefaultTreeModel model;

	TNode root;

	boolean ignoreSelectionEvent = false;

	OptionTree() {
	    super(new DefaultTreeModel(new TNode("root")));
	    model = (DefaultTreeModel)getModel();
	    root = (TNode)model.getRoot();

	    setEditable(false);
	    getSelectionModel()
		.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
	    addTreeSelectionListener(this);

	    setShowsRootHandles(true);
	    setRootVisible(false);

	    putClientProperty("JTree.lineStyle", "Angled");
	    DefaultTreeCellRenderer tcr = new DefaultTreeCellRenderer();
	    tcr.setOpenIcon(null);
	    tcr.setClosedIcon(null);
	    tcr.setLeafIcon(null);
	    setCellRenderer(tcr);

	    // Remember that much may have been done before this
	    // tool was created.  So we have to ask the option-manager
	    // what options already exist.
	    populateTree();
	}

	void reset() {
	    root.removeAllChildren();
	    model.setRoot(root); // will tell the JTree of the change 
	}

	TNode findTNode(String name) {
	    for (Enumeration e = root.preorderEnumeration()
		     ; e.hasMoreElements(); ) {
		TNode n = (TNode)e.nextElement();
		if (n.getUserObject().equals(name))
		    return n;
	    }
	    return null;
	}

	TNode requireTNode(String name) {
	    TNode n = findTNode(name);
	    if (n != null)
		return n;
	    else
		throw new IllegalArgumentException
		    ("Cannot find " + name);
	}

	void populateTree() {
	    // /\/: We assume parents appear in the name-to-option map
	    // before their children.  Any children that appear before
	    // their parent will be put at the top level.
	    TNode parent = root;
	    for (Iterator i = optMan.getNameToOptionMap().keySet().iterator()
		     ; i.hasNext();) {
		String name = (String)i.next();
		String parentName = parentOptionName(name);
		TNode n = new TNode(name);
		if (parentName == null)
		    parent = root;
		else if (parent.getUserObject().equals(parentName))
		    ; // Already have the right parent
		else {
		    parent = findTNode(parentName);
		    if (parent == null)
			parent = root;
		}
		parent.add(n);
	    }
	    model.nodeStructureChanged(root);
	    // /\/: Now we have to force everything to be expanded.
	    for (Enumeration e = root.preorderEnumeration()
		     ; e.hasMoreElements(); ) {
		TNode n = (TNode)e.nextElement();
		expandPath(n.getTreePath());
	    }
	    // Indicate the current option
	    TNode current = requireTNode(optMan.getOption().getName());
	    TreePath path = current.getTreePath();
	    setSelectionPath(path);
	    scrollPathToVisible(path);
	}

	private String parentOptionName(String name) {
	    // /\/: For now.  Really, we should keep track of
	    // the actual parent rather than go by names.
	    IPlanOptionManager.Opt opt = optMan.getOption(name);
	    try {
		String parentName = optMan.parentName(opt.getName());
		IPlanOptionManager.Opt parent = optMan.getOption(parentName);
		return parent.getName();
	    }
	    catch (Exception e) {
		Debug.noteln("Couldn't find parent option name for " + name +
			     " because: " + Debug.describeException(e));
		return null;
	    }
	}

	/* Method for TreeSelectionListener. */

	public void valueChanged(TreeSelectionEvent e) {
	    if (ignoreSelectionEvent)
		return;
	    TNode n = (TNode)getLastSelectedPathComponent();
	    if (n == null)
		return;
	    Debug.noteln("Option tree selection", n.getTreePath());
	    optMan.setOption((String)n.getUserObject());
	}

	/* OptionListener methods */

	public void optionAdded(OptionEvent e) {
	    String name = e.getOptionName();
	    TNode n = new TNode(name);
	    TNode parent = parentTNode(name);
	    model.insertNodeInto(n, parent, parent.insertionIndex(n));
	    scrollPathToVisible(n.getTreePath());
	}

	private TNode parentTNode(String name) {
	    String parentName = parentOptionName(name);
	    if (parentName == null)
		return root;
	    else {
		TNode parent = findTNode(parentName);
		return parent == null ? root : parent;
	    }
	}

	public void optionSet(OptionEvent e) {
	    TNode n = requireTNode(e.getOptionName());
	    try {
		ignoreSelectionEvent = true;
		setSelectionPath(n.getTreePath());
	    }
	    finally {
		ignoreSelectionEvent = false;
	    }
	}

	public void optionRenamed(OptionEvent e, String oldName) {
	    root.removeAllChildren();
	    populateTree();
	}

	public void optionContentsChanged(OptionEvent e, EventObject how) {
	    // We ignore this.
	}

	public void optionDeleted(OptionEvent event) {
	    TNode n = requireTNode(event.getOptionName());
	    // /\/: If the node has children, we have to put them
	    // back in the tree after removing the node.  We put
	    // them at the top level rather than find the youngest
	    // still undeleted ancestor.  That is less than ideal
	    // but consistent with what happens when new nodes
	    // are added.
	    model.removeNodeFromParent(n);
	    if (n.getChildCount() == 0)
		return;
	    TNode firstChild = (TNode)n.getFirstChild();
	    for (Enumeration e = n.children(); e.hasMoreElements();) {
		TNode child = (TNode)e.nextElement();
		model.insertNodeInto(child, root, root.insertionIndex(n));
	    }
	    scrollPathToVisible(firstChild.getTreePath());
	}

	/* End of OptionListener methods */

    }

    class TNode extends DefaultMutableTreeNode {

	TNode(String userObject) {
	    super(userObject);
	}

	public TreePath getTreePath() {
	    return new TreePath(getPath());
	}

	int insertionIndex(TNode newChild) {
	    String newChildName = (String)newChild.getUserObject();
	    int index = 0;
	    for (Enumeration e = children(); e.hasMoreElements();) {
		TNode n = (TNode)e.nextElement();
		String nName = (String)n.getUserObject();
		if (nName.compareTo(newChildName) > 0)
		    break;
		index++;
	    }
	    return index;
	}

    }

    /*
     * Option comparison matrix
     */

    class OptionMatrix extends JEditorPane implements OptionListener {

	HTMLEditorKit matrixEditorKit = new IXHtmlEditorKit();

	Map urlToEvaluationMap = new HashMap();

	Gensym.Generator nameGen = new Gensym.Generator();

	OptionMatrix() {
	    // If we didn't want to use our own editor-kit, we could just do
	    // super("text/html", "");
	    super();
	    setEditorKitForContentType("text/html", matrixEditorKit);
	    setContentType("text/html");
	    setText("");
	    Debug.expectSame(matrixEditorKit, getEditorKit());

	    // It needs to be non-editable for hyperlinks to work.
	    setEditable(false);
	    addHyperlinkListener(new LinkListener());
	    addMouseListener(new LinkMouseListener());

	    populateMatrix();

	}

	void reset() {
	    setText("");
	    urlToEvaluationMap.clear();
	    populateMatrix();
	}

	void populateMatrix() {
	    Map optMap = optMan.getNameToOptionMap();
	    HtmlStringWriter html = new MatrixHtmlStringWriter();
	    urlToEvaluationMap.clear();
	    // Table
	    html.tag("table", "border=1 cellspacing=0");
	    html.newLine();
	    // Top row
	    html.tag("tr");
	    html.tagged("th", "Option:");
	    for (Iterator i = optMap.keySet().iterator(); i.hasNext();) {
		String name = (String)i.next();
		html.tagged("td", name);
	    }
	    html.end("tr");
	    html.newLine();
	    // Eval rows
	    // For each plan-evaluator
	evaluatorLoop:
	    for (Iterator ei = optMan.getPlanEvaluators().iterator()
		     ; ei.hasNext();) {
		PlanEvaluator eval = (PlanEvaluator)ei.next();
		if (!eval.isVisible())
		    continue evaluatorLoop;
		html.tag("tr");
		html.tagged("td", "align=right", eval.getShortDescription());
		// For each option ...
	    optionLoop:
		for (Iterator oi = optMap.values().iterator(); oi.hasNext();) {
		    IPlanOptionManager.Opt option = 
			(IPlanOptionManager.Opt)oi.next();
		    PlanEvaluation val = option.getPlanEvaluation(eval);
		    if (val == null) {
			html.tagged("td", ".");
			continue optionLoop;
		    }
		    Object v = val.getValue();
		    if (val.hasDetails())
			html.tagged("td", makeDetailsLink(val));
		    else
			html.tagged("td", v.toString());
		}
		html.end("tr");
		html.newLine();
	    }
	    // End table
	    html.end("table");
	    html.newLine();
	    setText(html.toString());
	}

	String makeDetailsLink(PlanEvaluation val) {
	    return "<a href=\"" + makeDetailsURL(val) + "\">"
		+  val.getValue()
		+  "</a>";
	}

	String makeDetailsURL(PlanEvaluation val) {
	    String url = "http://" + nameGen.nextString("val");
	    urlToEvaluationMap.put(url, val);
	    return url;
	}

	class LinkListener implements HyperlinkListener {
	    LinkListener() { }
  	    public void hyperlinkUpdate(HyperlinkEvent e) {
		try {
		    do_hyperlinkUpdate(e);
		}
		catch (Throwable t) {
		    Debug.displayException(t);
		}
	    }
  	    public void do_hyperlinkUpdate(HyperlinkEvent e) {
		if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
		    // JEditorPane pane = (JEditorPane)e.getSource();
		    // HTMLDocument doc = (HTMLDocument)pane.getDocument();
		    String url = e.getURL().toString();
		    Debug.noteln("Click on", url);
		    PlanEvaluation val =
			(PlanEvaluation)urlToEvaluationMap.get(url);
		    Debug.expect(val != null,
				 "Can't find plan-evaluation for", url);
		    Object oldValue = val.getValue();
		    val.detailsRequested();
		    if (val.getValue() != oldValue)
			populateMatrix();
		}
	    }
	}

	class LinkMouseListener extends MouseAdapter {
	    public void mouseClicked(MouseEvent e) {
		// Debug.noteln("Mouse clicked", e);
	    }
	}

	/* OptionListener methods */

	public void optionAdded(OptionEvent e) {
	    populateMatrix();
	}

	public void optionSet(OptionEvent e) {
	    populateMatrix();
	}

	public void optionRenamed(OptionEvent e, String oldName) {
	    populateMatrix();
	}

	public void optionContentsChanged(OptionEvent e, EventObject how) {
	    populateMatrix();
	}

	public void optionDeleted(OptionEvent e) {
	    populateMatrix();
	}

	/* End of OptionListener methods */

    }

    static class MatrixHtmlStringWriter extends HtmlStringWriter {

	public MatrixHtmlStringWriter() {
	    super();
	    setDefaultAttributes("tr", "align=center");
	    setDefaultAttributes("th", "align=right bgcolor=\"#99ccff\"");
	}

    }

}

// Issues:
// * Perhaps the GUI shouldn't work with (and shouldn't be able to see)
//   the option objects.

