/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 19 01:59:21 2003 by Jeff Dalton
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 */

package ix.ip2.test;

import java.util.*;

import ix.icore.*;
import ix.ichat.ChatMessage;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;		// for debugging output

/**
 * A test that sends a series of messages generated from a TestItem template.
 * Either the same message can be sent to a series of different agents,
 * or a given agent can be sent a series of messages of the same type
 * which vary in their "main content" (= pattern for Issues and Activities,
 * text for Reports and ChatMessages).  If greater variety is required,
 * a TestSequence may be specified directly.
 *
 * <p>A delay in milliseconds may be placed before each message.
 * By default, this is taken from getDelayBefore() of the template,
 * but different initial and between delays may be specified instead.
 *
 * @see TestSequence
 */
public class TestSequenceGenerator extends PanelTest {

    protected static final int UNSPECIFIED = -1;

    protected int initialDelay = UNSPECIFIED;
    protected int delayBetween = UNSPECIFIED;
    protected TestItem template;
    protected List toNames;
    protected List contentStrings;

    public TestSequenceGenerator() {
        super();
    }

    public int getInitialDelay() {
        return initialDelay;
    }

    public void setInitialDelay(int initialDelay) {
        this.initialDelay = initialDelay;
    }

    public int getDelayBetween() {
        return delayBetween;
    }

    public void setDelayBetween(int delayBetween) {
        this.delayBetween = delayBetween;
    }

    public TestItem getTemplate() {
        return template;
    }

    public void setTemplate(TestItem template) {
        this.template = template;
    }

    public List getToNames() {
        return toNames;
    }

    public void setToNames(List toNames) {
        this.toNames = toNames;
    }

    public List getContentStrings() {
        return contentStrings;
    }

    public void setContentStrings(List contentStrings) {
        this.contentStrings = contentStrings;
    }

    public List getTestItems() {
	return generateTestSequence().getTestItems();
    }

    public TestSequence generateTestSequence() {
	TestSequence seq = new TestSequence();
	List items = new LinkedList();

	// Generate the TestItems.
	if (toNames != null && contentStrings != null)
	    throw new IllegalArgumentException
		("Test sequence cannot be generated " +
		 "from both names and content strings.");
	else if (toNames != null)
	    generateForNames(toNames, items);
	else if (contentStrings != null)
	    generateForContents(contentStrings, items);
	else
	    throw new IllegalArgumentException
		("One of names or content strings must be specified " +
		 "for test sequence");

	// Put in the initial and between delays, if specified;
	// otherwise just leave the delays taken from the template.
	Iterator i = items.iterator();
	if (i.hasNext()) {
	    // Always takes the first item if it exists.
	    TestItem first = (TestItem)i.next();
	    if (initialDelay != UNSPECIFIED)
		first.setDelayBefore(initialDelay);
	}
	if (delayBetween != UNSPECIFIED) {
	    // First item, if any, has already been taken.
	    while (i.hasNext()) {
		TestItem item = (TestItem)i.next();
		item.setDelayBefore(delayBetween);
	    }
	}

	// Finish up.
	seq.setMenuText(menuText);
	seq.setTestItems(items);
	Debug.noteln("Generated sequence:", XML.objectToXMLString(seq));
	return seq;
    }

    protected void generateForNames(List toNames, List items) {
	for (Iterator i = toNames.iterator(); i.hasNext();) {
	    String toName = (String)i.next();
	    TestItem item = copyTemplate();
	    item.setToName(toName);
	    items.add(item);
	}
    }

    protected void generateForContents(List contentStrings, List items) {
	for (Iterator i = contentStrings.iterator(); i.hasNext();) {
	    String contentString = (String)i.next();
	    Debug.noteln("String", contentString);
	    TestItem item = copyTemplate();
	    setContent(item.getContents(), contentString);
	    items.add(item);
	}
    }

    protected TestItem copyTemplate() {
	TestItem copy = new TestItem();
	copy.setDelayBefore(template.getDelayBefore());
	copy.setToName(template.getToName());
	copy.setContents((Sendable)Util.clone(template.getContents()));
	return copy;
    }

    protected void setContent(Sendable toSend, String text) {
	if (toSend instanceof TaskItem) {
	    TaskItem ti = (TaskItem)toSend;
	    LList pattern = PatternParser.parse(text);
	    ti.setPattern(pattern);
	}
	else if (toSend instanceof Report) {
	    Report report = (Report)toSend;
	    report.setText(text);
	}
	else if (toSend instanceof ChatMessage) {
	    ChatMessage message = (ChatMessage)toSend;
	    message.setText(text);
	}
	else
	    throw new IllegalArgumentException
		("Cannot set contents of " + toSend);
    }

}
