/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jun  9 01:10:57 2008 by Jeff Dalton
 * Copyright: (c) 2007, 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.Constraint;
import ix.icore.domain.End;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

/**
 * Strictly-consumable resource usage manager.
 * /\/: This is a very simple version.
 */
public class Scrum implements ConstraintManager {

    private static final Symbol
	CONSUME = Symbol.intern("consume"),
	OVERALL = Symbol.intern("overall");

    // Maps a resource pattern (an LList) to a context-dependent
    // map from PNodes to Longs, where each Long is the number
    // of resource units still available at that node.
    Map<LList,ContextHashMap> resourceTable =
	new HashMap<LList,ContextHashMap>();

    public Scrum() {
    }

    public void registerWith(ConstraintManagerRegistry r) {
	r.addConstraintManager
	    (this, Symbol.intern("resource"),
	           new Symbol[]{CONSUME, OVERALL});
    }

    public void reset() {
	resourceTable.clear();
    }

    public void clear() {
	for (ContextHashMap avail: resourceTable.values()) {
	    avail.clear();
	}
    }

    public boolean mightInteract(PNodeEnd ne1, Constraint c1,
				 PNodeEnd ne2, Constraint c2) {
	return false;		// order doesn't matter
    }

    /** @throws UnsupportedOperationException if called. */
    public void addConstraint(Constraint c) {
      throw new UnsupportedOperationException
	("Method addConstraint(Constraint) is not supported by this CM.");
    }

    public void addConstraint(PNode node, Constraint c) {
	Debug.noteln("SCRUM adding", c);
	Debug.noteln("At", node);
	// evalConstraint(node, c);
    }

    protected void evalConstraint(PNode node, Constraint c) {
	c = (Constraint)Variable.removeAllVars(c);
	Symbol r = c.getRelation();
	LList pattern = c.getPattern();
	Long value = Util.mustBe(Long.class, c.getValue());
	long v = value.longValue();
	ContextHashMap avail = getAvailabilityMap(pattern);
	if (r == CONSUME) {
	    consume(avail, pattern, node, v);
	}
	else if (r == OVERALL) {
	    // Consume v units but also install v as a limit on what
	    // descendents of the node can consume.
	    consume(avail, pattern, node, v);
	    avail.put(node, new Long(v));
	}
	else
	    Debug.warn("Invalid constraint: " + c);
    }

    public void evalAtBegin(PNodeEnd ne, Constraint c) {
	Debug.noteln("SCRUM eval " + c + " at " + ne);
	Debug.expectSame(End.BEGIN, ne.getEnd());
	evalConstraint(ne.getNode(), c);
    }

    public void evalAtEnd(PNodeEnd ne, Constraint c) {
	Debug.noteln("SCRUM eval " + c + " at " + ne);
	Debug.expectSame(End.END, ne.getEnd());
    }

    protected ContextHashMap getAvailabilityMap(LList resourcePattern) {
	ContextHashMap avail = resourceTable.get(resourcePattern);
	if (avail == null)
	    resourceTable.put(resourcePattern, avail = new ContextHashMap());
	return avail;
    }

    protected void consume(ContextHashMap avail,
			   LList pattern, 
			   PNode node, long v) {
	for (PNode at = node; at != null; at = at.getParentPNode()) {
	    Long remaining = (Long)avail.get(at);
	    if (remaining != null) {
		long r = remaining.longValue();
		r -= v;
		Debug.noteln("SCRUM: Consuming " + v + " at " + at +
			     " for " + node + ", leaving " + r);
		if (r >= 0) {
		    avail.put(at, new Long(r));
		    return;
		}
		else
		    throw new FailureException
			("Less than " + v + " " + pattern +
			 " available to " + node);
	    }
	}
    }

}

// Issues:
// * Should ConstraintAssociator implement ConstraintManager?

