/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Nov 13 17:13:22 2004 by Jeff Dalton
 * Copyright: (c) 2003 - 2005, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.plan.Plan;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.XML;

/**
 * Generates a {@link HandlerAction} that loads a plan.
 */
public class LoadPlanHandler extends ActivityHandler {

    public static final Symbol S_LOAD_PLAN = Symbol.intern("load-plan");

    protected Ip2 ip2;

    Symbol verb;

    public LoadPlanHandler(Ip2 ip2) {
	this(ip2, "load-plan");
    }

    public LoadPlanHandler(Ip2 ip2, String verb) {
	super(Strings.capitalize(Strings.replace("-", " ", verb)));
	this.ip2 = ip2;
	this.verb = Symbol.intern(verb);
    }

    public List getSyntaxList() {
	return (LList)Lisp.readFromString
	    ("((" + verb + " ?plan))");
    }

    public boolean appliesTo(AgendaItem item) {
	LList pattern = item.getPattern();
	return pattern.length() == 2
	    && pattern.get(0) == verb;
    }
	
    public void addHandlerActions(AgendaItem item) {
	item.addAction(new LoadPlanAction(item));
	item.addAction(new ReplacePlanAction(item));
    }

    class LoadPlanAction extends HandlerAction {

	AgendaItem item;

	LoadPlanAction(AgendaItem item) {
	    this(item, "Load plan");
	}

	LoadPlanAction(AgendaItem item, String shortDescription) {
	    this.item = item;
	    this.shortDescription = shortDescription;
	}

	public boolean isReady() {
	    return Variable.isFullyBound(item.getPattern());
	}

	public ActionUnreadyReason getUnreadyReason() {
	    return new UnboundVariablesUnreadyReason(this, item);
	}

	public void handle(AgendaItem item) {
	    ip2.loadPlan(getSpecifiedPlan(item));
	    item.setStatus(Status.COMPLETE);
	}

	Plan getSpecifiedPlan(AgendaItem item) {
	    LList pattern = (LList)Variable.removeVars(item.getPattern());
	    Object spec = pattern.get(1);
	    if (spec instanceof Plan)
		return (Plan)spec;
	    else if (spec instanceof Symbol || spec instanceof String)
		return (Plan)XML.readObject(Plan.class, spec.toString());
	    else {
		throw new IllegalArgumentException
		    (spec + " is not a valid reference to a plan");
	    }
	}

    }

    class ReplacePlanAction extends LoadPlanAction {

	ReplacePlanAction(AgendaItem item) {
	    super(item, "Replace plan, keeping state");
	}

	public void handle(AgendaItem item) {
	    // First, make sure we can actually get the plan.
	    Plan plan = getSpecifiedPlan(item);
	    // Clear out everything but the world state.
	    ip2.resetViewers();
	    ip2.clearAllButState();
	    // Load the plan.
	    ip2.loadPlan(plan);
	    // The item was removed by the reset, so we don't do
	    // item.setStatus(Status.COMPLETE);
	}

    }

}
