/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr  7 23:10:27 2005 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

// /\/: The division of labour between this and the model manager
// still isn't right.

/**
 * Stores the current world state and certain things about it.
 */
public class Ip2WorldStateManager {

    public static final Symbol
	S_WORLD_STATE = Symbol.intern("world-state"),
	S_CONDITION   = Symbol.intern("condition"),
	S_EFFECT      = Symbol.intern("effect");

    protected Ip2ModelManager modelManager;

    protected ContextMultiMap nodeToConditions = makeNodeConstraintsMap();
    protected ContextMultiMap nodeToEffects = makeNodeConstraintsMap();
    protected ContextMap worldStateMap = makeWorldStateMap(); // p -> v

    public Ip2WorldStateManager(Ip2ModelManager modelManager) {
	this.modelManager = modelManager;
    }

    public void reset() {
	nodeToConditions.clearCompletely();
	nodeToEffects.clearCompletely();
	worldStateMap.clearCompletely();
    }

    public void clear() {
	nodeToConditions.clear();
	nodeToEffects.clear();
	worldStateMap.clear();
    }

    protected ContextMap makeWorldStateMap() {
	return new ContextHashMap();
    }

    protected ContextMultiMap makeNodeConstraintsMap() {
	return new ContextMultiHashMap();
    }

    public ListOfConstraint getNodeConditions(PNode node) {
	// /\/: Inefficient to make a new linked-list.
	List conds = (List)nodeToConditions.get(node);
	if (conds != null)
	    return new LinkedListOfConstraint(conds);
	else
	    return new LinkedListOfConstraint();
    }

    public List getNodeEffects(PNode node) {
	return (List)nodeToEffects.get(node);
    }

    public Map getWorldStateMap() {	// for PlanMaker /\/
	return worldStateMap;
    }

    public Object getPatternValue(LList pattern) {
	return worldStateMap.get(pattern);
    }

    public void addConstraint(PNode node, Constraint c) {
	Symbol type = c.getType();
	Symbol relation = c.getRelation();
	Debug.expect(type == S_WORLD_STATE);
	PatternAssignment pv = (PatternAssignment)c.getParameter(0);
	if (relation == S_CONDITION) {
	    nodeToConditions.addValue(node, c);
	}
	else if (relation == S_EFFECT) {
	    nodeToEffects.addValue(node, pv); // pv, not c /\/
	}
	else {
	    throw new ConsistencyException
		("Unexpected constraint " + c);
	}
    }

    public Map handleEffects(PNode node, List effects) {
	// N.B. All Variables in the effects must be bound.
	// /\/: Takes a list of PatternAssignments, NOT a list of Constraints.
	Map delta = new StableHashMap();
	for (Iterator i = effects.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    Debug.noteln("Effect", pv);
	    LList pattern = pv.getPattern();
	    Object value = pv.getValue();
	    Debug.expect(Variable.isFullyBound(pattern));
	    Debug.expect(Variable.isFullyBound(value));
	    LList p = (LList)Variable.removeVars(pattern);
	    Object v = Variable.removeVars(value);
	    delta.put(p, v);
	    assign(p, v, node);
	}
	return delta;
    }

    public Map handleEffects(List effects) {
	return handleEffects(null, effects);
    }

    public void deleteEffect(PatternAssignment pv) {
	Set vars = Variable.varsAnywhereIn(pv);
	Set unbound = Variable.unboundVarsIn(vars);
	if (!unbound.isEmpty())
	    throw new IllegalArgumentException
		("Can't delete effect that is not fully bound: " + pv);
	Object currentValue = getPatternValue(pv.getPattern());
	if (!pv.getValue().equals(currentValue))
	    throw new IllegalArgumentException
		("Can't delete effect " + pv + " when the current " +
		 "value of the pattern, " + currentValue + ", is " +
		 "different.");
	worldStateMap.remove(pv.getPattern());
	Debug.expect(getPatternValue(pv.getPattern()) == null);
    }

    protected void assign(LList p, Object v, PNode at) {
	worldStateMap.put(p, v);
    }

}
