/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Oct 25 13:41:30 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;
import javax.imageio.ImageIO;

import java.awt.AWTException;
import java.awt.Component;
import java.awt.Frame;
import java.awt.Graphics2D;
import java.awt.Robot;
import java.awt.Rectangle;

import java.awt.image.BufferedImage;
import java.awt.event.*;

import java.io.File;
import java.io.IOException;

import java.util.*;

import ix.icore.IXAgentExtension;

import ix.iface.util.ToolController;
import ix.iface.util.IFUtil;

import ix.util.*;

public class FrameSnapshotExtension implements IXAgentExtension {

    private String imageDirectory = "/tmp";
    private String imageType = "png";        // can be png or jpg

    private Ip2 ip2;

    public FrameSnapshotExtension(Ip2 ip2) {
	this.ip2 = ip2;
    }

    public void installExtension() {
	ip2.addTool(new ToolController("Snapshot Tool") {
	    public Object createTool() {
		return new SnapshotTool();
	    }
	});
    }

    private void snapAllFrames() {
	for (Frame f: Frame.getFrames()) {
	    if (isSnappable(f))
		savePicture((JFrame)f);
	}
    }

    private boolean isSnappable(Frame f) {
	return f instanceof JFrame
	    && f.isVisible()
	    && !(f instanceof SnapshotTool.SnapFrame);
    }

    private void savePicture(JFrame frame) {
	BufferedImage image = takePicture(frame);
	writeImageFile(image, imagePathname(frame));
    }

    private BufferedImage takePicture(JFrame frame) {

	// Get an image of the frame.
	// We use the content-pane, rather than the frame itself,
	// because if we use the frame there can be a black area
	// where the title bar would be.
	Component p = frame.getContentPane();
        BufferedImage image = 
	    new BufferedImage(p.getWidth(),
			      p.getHeight(),
			      BufferedImage.TYPE_INT_RGB);
        Graphics2D g2 = image.createGraphics();
	try {
	    p.paintAll(g2);
	}
	finally {
	    g2.dispose();
	}
	return image;
    }

    private void writeImageFile(BufferedImage image, String fileName) {

	// Write it as a file of type imageType, usually "png".
	Debug.noteln("Writing", fileName);
	try {
	    ImageIO.write(image, imageType, new File(fileName));
	}
	catch (IOException e) {
	    throw new RethrownException(e);
	}

    }

    private String imagePathname(Frame f) {
	String slash = System.getProperty("file.separator");
	return imageDirectory + slash + fileName(f) + ".png";
    }

    /** 
     * The file-name, without directory or type, that corresponds
     * to the frame.
     */
    private String fileName(Frame f) {
	// Replace each space by a "-".
	return Strings.replace(" ", "-", f.getTitle());
    }

    /**
     * The GUI that lets the user ask for pictures.
     * It has a "Snap" button that the user should press.
     */
    protected class SnapshotTool {

	private JFrame frame;

	public SnapshotTool() {
	    frame = new SnapFrame();
	    frame.getContentPane()
		     .add(IFUtil.makeButton("Snap", new SnapListener()));
	    frame.pack();
	}

	public void setVisible(boolean v) {
	    frame.setVisible(v);
	}

	private class SnapFrame extends JFrame {
	    private SnapFrame() {
		super(ip2.getAgentDisplayName() + " Snap");
	    }
	}

	private class SnapListener implements ActionListener {
	    public void actionPerformed(ActionEvent event) {
		// takePictureFromFrame();
		snapAllFrames();
	    }
	}

    }

}
