/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 14 04:06:11 2007 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.ip2.*;
import ix.ip2.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.icore.process.ProcessModelManager;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Satisfies conditions
 */
public class ConditionHandler extends ActivityHandler {

    protected Ip2 ip2;
    protected Ip2ModelManager modelManager;

    public ConditionHandler(Ip2 ip2) {
	super("Satisfy conditions");
	this.ip2 = ip2;
	this.modelManager = (Ip2ModelManager)ip2.getModelManager();
    }

    public boolean appliesTo(AgendaItem item) {
	// return item.isExpanded() && item.getStatus() == Status.POSSIBLE;
	return true;
    }

    public void addHandlerActions(AgendaItem item) {
	if (isUsefulFor(item))
	    item.addAction(new SatisfyAction(item));
	else
	    item.addItemListener(new ItemListener());
    }

    protected boolean isUsefulFor(AgendaItem item) {
	return item.isExpanded() && item.getHandledBy() == null
	    && !Collect.isEmpty(modelManager.getNodeConditions(item));
    }


    protected class ItemListener extends AgendaItemAdapter {
	public ItemListener() { }
	public void agendaItemEdited(AgendaItemEvent e) {
	    AgendaItem item = (AgendaItem)e.getSource();
	    // The item may have been expanded, so that adding
	    // a SatisfyAction now makes sense.
	    if (isUsefulFor(item))
		ip2.getController()
		    .reconsiderHandler(ConditionHandler.this, e);
	}
    }

    public void reviseHandlerActions(AgendaItem item, Object reason) {
	if (item.findAction(SatisfyAction.class) == null)
	    addHandlerActions(item);
    }

    class SatisfyAction extends HandlerAction {

	AgendaItem item;
	MatchEnv env = SimpleMatcher.emptyEnv;
	ListOfConstraint filters;
	List filterEnvs = Collections.EMPTY_LIST;

	Set unboundVariables;

	SatisfyAction(AgendaItem item) {
	    this.shortDescription = "Satisfy conditions";
	    this.item = item;
	    this.filters = modelManager.getNodeConditions(item);
	}

	public boolean isReady() {
	    return filters.isEmpty() || !filterEnvs.isEmpty();
	}

	public void computeStatus() {
	    if (!filters.isEmpty())
		filterEnvs = modelManager.evalFilters(filters, env);
	}

	public ActionUnreadyReason getUnreadyReason() {
	    List satisfiable = modelManager.testFilters(filters, env);
	    return new UnsatisfiedConditionsUnreadyReason
		(this, filters, satisfiable, env);
	}

	public void handle(AgendaItem item) {

	    Debug.expect(item == this.item);

	    // Get filter conds from the instantiated refinement.
	    // List conds = item.getRefinement().getFilterConditions();
	    ListOfConstraint conds = filters;

	    // Don't need to pass an env this time.
	    // The model manager returns the live branches.
	    List branches = modelManager.reevaluateFilters(conds);
	    Debug.expect(!branches.isEmpty(),
			 "Cannot match filter conditions " + conds);

	    // The branches tell us all the ways we could satisfy
	    // all the filters by binding.  Each branch is a way to
	    // satisy all the filters.

	    Set condVars = Variable.varsAnywhereIn(conds);
	    Set unboundVars = Variable.unboundVarsIn(condVars);

	    Debug.noteln("Cond vars", condVars);
	    Debug.noteln("Unbound", unboundVars);

	    if (unboundVars.isEmpty()) {
		//modelManager.satisfyConds(item, conds, new MatchEnv());
		adjustStatus(item);
		return;
	    }

	    // We now have to get the user to bind all the unbound
	    // vars from the conditions, and then we can satisfy
	    // the conds.

	    unboundVariables = unboundVars;  // remember
	    BindingViewer binder =
		new BindingViewer(ip2, unboundVars) {
		    public void handleBindings(Map newBindings) {
			if (newBindings != null) {
			    bind(newBindings);
			}
		    }
		};
	}

	protected void bind(Map newBindings) {
	    try {
		modelManager.bindVariables(newBindings);
	    }
	    catch (IllegalArgumentException e) {
		// The user might have picked incompatible values
		// when selecting for more than one variable.
		Debug.displayException(e);
	    }
	    Set stillUnbound = Variable.unboundVarsIn(unboundVariables);
	    Debug.noteln("Still unbound", stillUnbound);
	    if (stillUnbound.isEmpty()) {
		//modelManager.satisfyConds(item, filters, new MatchEnv());
		adjustStatus(item);
	    }
	    else {
		// /\/: User quit before binding enough vars
		item.setHandledBy(null);
	    }
	}

	protected void adjustStatus(AgendaItem item) {
	    item.setStatus(Collect.isEmpty(item.getChildren())
  			   ? Status.COMPLETE : Status.EXECUTING);
	}

    }

}
