/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Dec  9 18:02:34 2008 by Jeff Dalton
 * Copyright: (c) 2001, 2002, 2006 - 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;
import java.awt.event.*;
import java.util.*;

import ix.icore.*;
import ix.icore.plan.Plan;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.XMLSaver;

import ix.iface.util.CatchingActionListener;

/**
 * Provides the shell of a popup menu for agenda items in an agenda
 * viewer.  The viewer defines a subclass that fills in the abstract
 * methods.
 */
public abstract class AbstractAgendaItemPopupMenu extends JPopupMenu
       implements ActionListener {

    protected Ip2 ip2;
    protected AgendaItem item;

    protected Set unboundVariables;

    protected JMenuItem foldItem = makeMenuItem("Fold");
    protected JMenuItem bindItem = makeMenuItem("Bind Variables");
    protected JMenuItem deleteItem = makeMenuItem("Delete");
    protected JMenuItem completeItem = makeMenuItem("Consider Done");

    protected JMenu insertMenu = new JMenu("Insert");
    protected JMenuItem insertBeforeItem = makeMenuItem("Before");
    protected JMenuItem insertAfterItem = makeMenuItem("After");

    protected JMenuItem forwardReportItem = null;

    public AbstractAgendaItemPopupMenu(Ip2 ip2, AgendaItem item) {
	super();
	this.ip2 = ip2;
	this.item = item;

	add(makeMenuItem("Show Details"));
	add(foldItem);
	add(bindItem);

	// Note that variables might be added as a result of expansion,
	// which is why we don't decide whether to enable the bindItem
	// until the noticeItemState method is called.

	foldItem.setEnabled(false);
	bindItem.setEnabled(false);

	add(deleteItem);
	deleteItem.setEnabled(false);

	insertMenu.add(insertBeforeItem);
	insertMenu.add(insertAfterItem);
	if (item instanceof ActivityItem)
	    add(insertMenu);	// only for activities for now /\/

	add(makeMenuItem("Copy to Messenger"));

	if (item.getAbout().getReportBack() == YesNo.YES)
	    add(makeMenuItem("Send Report to " +
			     item.getAbout().getSenderId()));

        if (item instanceof ActivityItem) {
            add(makeMenuItem("Send Subplan"));
            add(makeMenuItem("Send Subplan With Report-Back"));
            add(makeMenuItem("Save Subplan"));
        }

    }

    protected JMenuItem makeMenuItem(String text) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(this));
	return item;
    }

    public void noticeItemState() {
	foldItem.setEnabled(!item.getChildren().isEmpty());
	foldItem.setText(isOpen() ? "Fold" : "Unfold");
	bindItem.setEnabled(!item.getUnboundVars().isEmpty());
	deleteItem.setEnabled(canDelete(item));

	if (item.getStatus() == Status.EXECUTING
	      || item.getStatus() == Status.COMPLETE)
	    insertBeforeItem.setEnabled(false);

	// See if the item has been forwarded to another agent,
	// and if so add an item that will send a report to that agent.
	Name forwardedTo = item.getAbout().getForwardedTo();
	if (forwardedTo != null && forwardReportItem == null) {
	    forwardReportItem = makeMenuItem("Send Report to " + forwardedTo);
	    add(forwardReportItem);
	}

	// Let the main GUI contribute anything it wants to.
	ip2.getFrame().adjustAgendaItemPopup(this, item);

	// See if we might want to force the status to COMPLETE.
	// This matters because we might otherwise be stuck waiting
	// for e.g. a completion report from another agent that
	// might never arrive.
	if ((item.getStatus() == Status.EXECUTING ||
	     item.getStatus() == Status.IMPOSSIBLE)
	        && item.getChildren().isEmpty()) {
	    add(completeItem);
	}

    }

    protected boolean canDelete(AgendaItem item) {
	return item.getParent() == null
	    && item.getChildren().isEmpty()
	    && (item.getStatus() == Status.POSSIBLE
		|| item.getStatus() == Status.BLANK);
    }

    // Methods that subclasses must define

    /**
     * Tell the viewer to put up a details window about this menu's
     * AgendaItem.
     */
    abstract void showDetails();

    /**
     * Returns true if the subitems of this menu's AgendaItem
     * are displayed and false if they are not.
     */
    abstract boolean isOpen();

    /**
     * Tell the view to hide the subitems of this menu's AgendaItem.
     */
    abstract void fold();

    /**
     * Tell the view to show the subitems of this menu's AgendaItem.
     */
    abstract void unfold();

    /**
     * Action interpreter for this menu's menu-items.
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("Item popup command", command);
	if (command.equals("Show Details")) {
	    showDetails();
	}
	else if (command.equals("Fold")) {
	    fold();
	}
	else if (command.equals("Unfold")) {
	    unfold();
	}
	else if (command.equals("Bind Variables")) {
	    Set unboundVars = item.getUnboundVars();
	    Debug.expect(!unboundVars.isEmpty(), "No variables to bind.");
	    unboundVariables = unboundVars;  // remember
	    BindingViewer binder =
		new BindingViewer(ip2, unboundVars) {
		    public void handleBindings(Map newBindings) {
			if (newBindings != null) {
			    bind(newBindings);
			}
		    }
		};
	}
	else if (command.equals("Delete")) {
	    if (Util.dialogConfirms(ip2.getFrame(),
                                    "Are you sure you want to delete?"))
		deleteItem();
	}
	else if (command.equals("Before")) {
	    insertBefore();
	}
	else if (command.equals("After")) {
	    insertAfter();
	}
	else if (command.equals("Copy to Messenger")) {
	    copyItemToMessenger();
	}
	else if (command.startsWith("Send Report")) {
	    if (e.getSource() == forwardReportItem) {
		// Send a report to the agent we gave a copy of the item.
		ip2.frame.getChatFrameVisible()
		    .initForwardReport(item.getAbout());
	    }
	    else {
		// Send a report to the agent that sent us the item.
		ip2.frame.getChatFrameVisible()
		    .initReportBackReport(item.getAbout());
	    }
	}
        else if (command.equals("Send Subplan")) {
            ActivityItem act = (ActivityItem)item;
            Plan plan = new PlanMaker(ip2).getPlan(act);
	    Activity activity =
		new Activity(Lisp.list(LoadPlanHandler.S_LOAD_PLAN,
				       plan));
	    ip2.frame.getSendPanelVisible().initActivity(activity);
        }
        else if (command.equals("Send Subplan With Report-Back")) {
            ActivityItem act = (ActivityItem)item;
            Plan plan = new PlanMaker(ip2).getPlanWithReportBack(act);
	    Activity activity =
		new Activity(Lisp.list(LoadPlanHandler.S_LOAD_PLAN,
				       plan));
	    ip2.frame.getSendPanelVisible().initActivity(activity);
        }
        else if (command.equals("Save Subplan")) {
            ActivityItem act = (ActivityItem)item;
            Plan plan = new PlanMaker(ip2).getPlan(act);
            new XMLSaver(ip2.getFrame(), Plan.class).saveObject(plan);
        }
	else if (command.equals("Consider Done")) {
	    item.setStatus(Status.COMPLETE);
	}
	else {
	    throw new UnsupportedOperationException
		("Nothing to do for item popup command " + command);
	}
    }

    protected void bind(Map newBindings) {
	Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	try {
	    mm.bindVariables(newBindings);
	}
	catch (IllegalArgumentException e) {
	    // The user might have picked incompatible values
	    // when selecting for more than one variable.
	    Debug.displayException(e);
	}
	// If we get here ...
	mm.logBindings(newBindings);
    }

    protected void deleteItem() {
	PanelController controller = ip2.getController();
	TaskItem about = item.getAbout();
	if (about instanceof Issue)
	    controller.getIssueAgenda().removeItem(item);
	else if (about instanceof Activity)
	    controller.getActivityAgenda().removeItem(item);
	else
	    Debug.expect(false, "Can't delete", about);
    }

    protected void copyItemToMessenger() {
	TaskItem about = item.getAbout();
	if (about instanceof Issue)
	    ip2.frame.getSendPanelVisible().initIssue((Issue)about);
	else if (about instanceof Activity)
	    ip2.frame.getSendPanelVisible().initActivity((Activity)about);    
	else
	    Debug.expect(false, "Can't copy to Messenger", about);
    }

    /*
     * Insert before / after
     */

    // /\/: This is still pretty ad hoc, and the popup menu
    // shouldn't be in charge of it.  Because new menus are
    // created when needed, we need a global place to keep
    // this insertion editor.

    protected static Map insertionEditors = new WeakHashMap();

    protected void insertBefore() {
	ensureInsertionEditor().insertBefore(item);
    }

    protected void insertAfter() {
	ensureInsertionEditor().insertAfter(item);
    }

    protected ActivityInsertionEditor ensureInsertionEditor() {
	if (insertionEditors.get(ip2) == null)
	    insertionEditors.put(ip2, new ActivityInsertionEditor(ip2));
	return (ActivityInsertionEditor)insertionEditors.get(ip2);
    }

}
