/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Mar 12 23:03:20 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2005, 2006, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.text.*;

import java.util.*;

import ix.util.*;

/**
 * A JTextPane for recording transcripts and similar information.
 */
public class TranscriptTextPane extends JTextPane {

    protected boolean foldLongLines = false;
    int excessLengthAllowed = 0;

    String NEWLINE = "\n";

    /**
     * Constructs an empty pane.
     */
    public TranscriptTextPane() {
	super();
	setEditable(false);
    }

    /**
     * Tells {@link #appendLine(String)} to break a long line
     * into separate lines that will fit in the display area.
     *
     * @param excessAllowed  the number of extra characters allowed
     *    before a line must be folded.
     */
    public void setFoldLongLines(int excessAllowed) {
	this.foldLongLines = true;
	this.excessLengthAllowed = excessAllowed;
    }

    /**
     * Adds the text at the end of the pane.  It should also
     * cause scrolling to the end.  Note that the other "append"
     * methods that don't involve color directly or indirectly
     * call this one.
     */
    public void append(String text) {
	SimpleAttributeSet at = new SimpleAttributeSet();
	StyleConstants.setForeground(at, Color.black);
	StyledDocument doc = getStyledDocument();
	try {
	    doc.insertString(doc.getLength(), text, at);
	}
	catch (BadLocationException e) {
	    Debug.displayException(e);
	}
    }

    /**
     * Adds the text at the end of the pane using the specified
     * color.  It should also cause scrolling to the end.  Note
     * that the other "append" methods that involve color directly
     * or indirectly call this one.
     */
    public void append(Color c, String text) {
	SimpleAttributeSet at = new SimpleAttributeSet();
	StyleConstants.setForeground(at, c);
	StyledDocument doc = getStyledDocument();
	try {
	    doc.insertString(doc.getLength(), text, at);
	}
	catch (BadLocationException e) {
	    Debug.displayException(e);
	}
    }

    /**
     * Appends the text, followed by a newline.
     */
    public void appendLine(String text) {
	if (foldLongLines) {
//  	    double width = getEffectiveWidth();
//  	    double charWidth = width / getColumnWidth();
//  	    charWidth -= 3;	// estimated scrollbar width
//  	    int maxLen = (int)charWidth + excessLengthAllowed;
	    int maxLen = 40;
	    Debug.noteln("maxLen =", maxLen);
	    if (text.length() > maxLen) {
		// This line might secretly contain newlines,
		// but that's handled by Strings.foldLongLines.
		append(Strings.foldLongLine(text, maxLen, "   ") + NEWLINE);
		return;
	    }
	}
	append(text + NEWLINE);
    }

    public void appendLine(Color c, String text) {
	if (foldLongLines) {
//  	    double width = getEffectiveWidth();
//  	    double charWidth = width / getColumnWidth();
//  	    charWidth -= 3;	// estimated scrollbar width
//  	    int maxLen = (int)charWidth + excessLengthAllowed;
	    int maxLen = 40;
	    Debug.noteln("maxLen =", maxLen);
	    if (text.length() > maxLen) {
		// This line might secretly contain newlines,
		// but that's handled by Strings.foldLongLines.
		append(c, Strings.foldLongLine(text, maxLen, "   ") + NEWLINE);
		return;
	    }
	}
	append(c, text + NEWLINE);
    }

    protected double getEffectiveWidth() {
	return getSize().getWidth();
    }

    /**
     * Breaks the text into lines, then appends the lines, each
     * prefixed by the specified number of spaces.
     */
    public void appendIndentedLines(int indent, String text) {
	Collection lines = Strings.breakIntoLines(text);
	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    appendLine(Strings.repeat(indent, " ") + line);
	}
    }

    /**
     * Breaks the text into lines, then appends the lines using
     * the specified color and with each line prefixed by the
     * specified number of spaces.
     */
    public void appendIndentedLines(int indent, Color color, String text) {
	Collection lines = Strings.breakIntoLines(text);
	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    appendLine(color, Strings.repeat(indent, " ") + line);
	}
    }

    public static void main(final String[] argv) {
        javax.swing.SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                do_main(argv);
            }
        });
    }

    private static void do_main(String[] argv) {
	JFrame frame = new JFrame();
	final TranscriptTextPane pane = new TranscriptTextPane();
	JTextField input = new JTextField(40) {
	    { enableEvents(AWTEvent.KEY_EVENT_MASK); }
	    protected void processKeyEvent(KeyEvent e) {
		super.processKeyEvent(e);
		if (e.getID() == KeyEvent.KEY_RELEASED) {
		    int key = e.getKeyCode();
		    if (key == '\n' || key == '\r') {
			String text = getText();
			pane.appendLine(text);
			setText("");
		    }
		}
	    }
	};
	frame.getContentPane().add(new JScrollPane(pane), BorderLayout.CENTER);
	frame.getContentPane().add(input, BorderLayout.SOUTH);
	frame.setSize(400,300);
	frame.setVisible(true);
    }

}
