/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 16 15:09:24 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, 2006, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import javax.swing.*;

import java.util.*;

import ix.util.*;

public abstract class SelectionPanel extends JPanel {

    JFrame frame;		// the frame that contains this panel
    String nameColTitle;
    String choiceColTitle;

    SelectionTable selectionTable;

    public SelectionPanel(JFrame frame,
			  String nameColTitle,
			  String choiceColTitle) {
	this.frame = frame;
	this.nameColTitle = nameColTitle;
	this.choiceColTitle = choiceColTitle;
	selectionTable = new SelectionTable();
	setupPanelContents();
    }

    void setupPanelContents() {
	// /\/: We could move things up and put an empty panel in the SOUTH
	// so that the real contents wouldn't be stretched out when in a
	// tabbed pane with something a different height, but that causes
	// other problems, such as no scrollbar appearing when the user
	// makes the frame too short.  Making this a VerticalPanel
	// and doing addFixedHeight(new JScrollPane(selectionTable))
	// also has the no-scrollbar problem.  However, a different
	// trick seems to work:
	setLayout(new BorderLayout());
//    	add(new JScrollPane(selectionTable), BorderLayout.CENTER);
	VerticalPanel p = new VerticalPanel();
	p.addFixedHeight(selectionTable);
	add(new JScrollPane(p), BorderLayout.CENTER);
	add(new ButtonPanel(), BorderLayout.SOUTH);
    }

    /**
     * Tells the panel to add rows corresponding to entries in
     * the Map returned by {@link #getCurrentValueMap()}.
     * This method should be called only when the table is empty.
     */
    protected void load() {
	selectionTable.loadTable();
    }

    /**
     * Removes everything from the table and then calls {@link #load()}.
     */
    protected void reload() {
	selectionTable.reloadTable();
    }

    /**
     * Inserts a row containing the specified name and value.
     * This method should be called only to install information
     * produced outside the panel; it is not called by the panel GUI.
     */
    protected void addTableEntry(String name, Object value) {
	new Row(name, value).insertInTable(selectionTable);
    }

    protected abstract ValueComboBox makeValueComboBox();

    /** Called when loading or relaoding the table. */
    protected abstract Map getCurrentValueMap();

    /** Called when the user commits to a deletion. */
    protected abstract void entryDeleted(String name);

    /** Called when the user commits to a change in value. */
    protected abstract void valueChanged(String name, Object value);

    /** Called when the user commits to adding a row. */
    protected abstract void entryAdded(String name, Object value);

    /**
     * The button panel in a SelectionPanel.
     */
    class ButtonPanel extends JPanel implements ActionListener {

	ButtonPanel() {
	    // Stick with default flowlayout for now.
	    add(makeButton("Commit"));
	    add(makeButton("Undo Uncommitted Changes"));
	}

	protected JButton makeButton(String command) {
	    JButton b = new JButton(command);
	    b.addActionListener
		(CatchingActionListener.listener(this));
	    return b;
	}

	/**
	 * Action interpreter
	 */
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Selectin panel action:", command);
	    if (command.equals("Commit")) {
		selectionTable.commitChanges();
	    }
	    else if (command.equals("Undo Uncommitted Changes")) {
		selectionTable.undoUncommittedChanges();
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

    }

    /**
     * The selection table in a SelectionPanel.
     */
    class SelectionTable extends JPanel {

	List rows = new LinkedList();

	GridColumn nameCol = new GridColumn(nameColTitle);
	GridColumn choiceCol = new GridColumn(choiceColTitle);

	Row newEntryRow;

	boolean ignoreReloads = false;

	SelectionTable() {
	    setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
	    setBorder(BorderFactory.createEtchedBorder());
	    add(nameCol);
	    add(Box.createHorizontalStrut(5));
	    add(choiceCol);
	}

	void loadTable() {
	    Map vm = getCurrentValueMap();
	    for (Iterator i = vm.keySet().iterator(); i.hasNext();) {
		String name = (String)i.next();
		Object value = vm.get(name);
		Row row = new Row(name, value);
		row.addToTable(this);
	    }
	    newEntryRow = new Row(new JTextField(40), makeValueComboBox());
	    newEntryRow.addToTable(this);
	}

	void reloadTable() {
	    if (ignoreReloads) {
		Debug.noteln("Selection table ignoring a reload");
		return;
	    }
	    nameCol.reset();
	    choiceCol.reset();
	    rows.clear();
	    loadTable();
	    invalidate();
	    frame.validate();
	    frame.pack();
	}

	void undoUncommittedChanges() {
	    for (Iterator ri = rows.iterator(); ri.hasNext();) {
		Row row = (Row)ri.next();
		row.reset();
	    }
	}

	void commitChanges() {

	    // Ignore reload requests while we do this
	    try {
		ignoreReloads = true;

		// Value changes?
		// If we allowed relaods, we'd have to iterate over a
		// copy of rows to prevent concurrent modification exceptions.
		for (Iterator ri = rows.iterator(); ri.hasNext();) {
		    Row row = (Row)ri.next();
		    if (row == newEntryRow)
			continue;
		    Object val = row.getSelectedValue();
		    if (val == null)
			SelectionPanel.this.entryDeleted(row.name);
		    else if (val != row.value) {
			SelectionPanel.this.valueChanged(row.name, val);
		    }
		}

		// Add an agent?
		String newName = newEntryRow.nameText.getText().trim();
		if (!newName.equals("")) {
		    Object val = 
			newEntryRow.getSelectedValue();
		    if (val != null)		 // ie not a "Delete"
			SelectionPanel.this.entryAdded(newName, val);
		}
	    }
	    finally {
		ignoreReloads = false;
	    }
	    reloadTable();
	}
    }

    /**
     * A row in the {@link SelectionPanel.SelectionTable}
     */
    class Row {

	String name;
	Object value;
	JTextField nameText;
	ValueComboBox valChoice;

	Row(String name, Object value) {
	    this.name = name;
	    this.value = value;
	    this.nameText = new JTextField(name);
	    this.valChoice = makeValueComboBox();

	    nameText.setEditable(false);
	    valChoice.setSelectedValue(value);
	}

	Row(JTextField nt, ValueComboBox vc) {
	    this.name = "";
	    this.value = vc.getSelectedValue();
	    this.nameText = nt;
	    this.valChoice = vc;
	}

	Object getSelectedValue() {
	    return valChoice.getSelectedValue();
	}

	void reset() {
	    if (nameText.isEditable()) nameText.setText(name);
	    valChoice.setSelectedValue(value);
	}

	void addToTable(SelectionTable t) {
	    t.rows.add(this);
	    t.nameCol.add(nameText);
	    t.choiceCol.add(valChoice);
	}

	void insertInTable(SelectionTable t) {
	    Debug.noteln("INserting", name);
	    Debug.noteln("ROWS", t.rows);
	    ListIterator ri = t.rows.listIterator();
	    while (ri.hasNext()) {
		Row row = (Row)ri.next();
		Debug.noteln("Comparing", row.name);
		// Insert before new-entry row (which is editable)
		// or before the first row with a lexicographically
		// later name.
		if (row.nameText.isEditable() 
		        || row.name.compareTo(name) > 0)
		    break;
	    }
	    int at = ri.nextIndex() - 1 + 1; 	// +1 because of col titles
	    t.rows.add(at - 1, this); 		// no title entry
	    t.nameCol.add(nameText, at);
	    t.choiceCol.add(valChoice, at);
	    t.invalidate();
	    frame.validate();
	    frame.pack();
	}

	public String toString() {
	    return "row[" + name + " --> " + getSelectedValue() + "]";
	}

    }

}
