/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Jun  4 18:33:03 2009 by Jeff Dalton
 * Copyright: (c) 2005, 2009, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.io.PrintWriter;
import java.io.Writer;
import java.io.OutputStream;
import java.io.OutputStreamWriter;

import java.util.*;

import ix.util.*;

/**
 * Simple support for HTML and similarly marked-up output.
 */
public class HtmlWriter {

    protected PrintWriter out;

    protected Map defaultAttributes = new HashMap();

    protected List tagStack = new ArrayList(100);

    public HtmlWriter(Writer w) {
	this.out = w instanceof PrintWriter
	    ? (PrintWriter)w
	    : new PrintWriter(w);
    }

    public HtmlWriter(OutputStream stream) {
	this(new OutputStreamWriter(stream));
    }

    public String getDefaultAttributes(String tag) {
	return (String)defaultAttributes.get(tag);
    }

    public void setDefaultAttributes(String tag, String attributes) {
	defaultAttributes.put(tag, attributes);
    }

    public void setDefaultAttributes(String tag, String[][] attributes) {
	StringBuffer at = new StringBuffer();
	for (int i = 0; i < attributes.length; i++) {
	    String key = attributes[i][0];
	    String value = attributes[i][1];
	    if (i > 0) at.append(" ");
	    at.append(key);
	    at.append("=\"");
	    at.append(value);
	    at.append("\"");
	}
	defaultAttributes.put(tag, at.toString());
    }

    public void flush() {
	out.flush();
    }

    public void close() {
	out.close();
    }

    public void newLine() {
	// out.println("");
	out.write("\n");
    }

    public void write(String text) {
	out.write(text);
    }

    public void writeln(String text) {
	out.println(text);
    }

    public List getTagStack() {
	return tagStack;
    }

    public int getTagDepth() {
	return tagStack.size();
    }

    public void indent(int depth) {
	for (int i = 0; i < depth; i++) {
	    out.write(" ");
	}
    }

    public void tagged(String tag, String contents) {
	tag(tag);
	write(contents);
	endTag(tag);
    }

    public void tagged(String tag, String attributes, String contents) {
	tag(tag, attributes);
	write(contents);
	endTag(tag);
    }

    public void tag(String tag) {
	tag(tag, getDefaultAttributes(tag));
    }

    public void tag(String tag, String attributes) {
	write("<");
	write(tag);
	if (attributes != null) {
	    write(" ");
	    write(attributes);
	}
	write(">");
	tagStack.add(tag);
    }

    public void tag(String tag, String[][] attributes) {
	write("<");
	write(tag);
        writeAttributes(attributes);
	write(">");
	tagStack.add(tag);
    }

    private void writeAttributes(String[][] attributes) {
        // N.B. The output starts with a space.
	for (int i = 0; i < attributes.length; i++) {
	    String key = attributes[i][0];
	    String value = attributes[i][1];
	    write(" ");
	    write(key);
	    write("=\"");
	    write(value);
	    write("\"");
	}
    }

    public void endTag(String tag) {
	if (tagStack.isEmpty())
	    throw new IllegalStateException("Missing <" + tag + "> tag.");
	String popped = (String)tagStack.remove(tagStack.size() - 1);
	if (!popped.equals(tag))
	    throw new IllegalStateException
		("Found <" + popped + "> instead of <" + tag + ">.");
	write("</");
	write(tag);
	write(">");
    }

    /* Empty-element tags */

    public void end(String tag) {
	endTag(tag);
    }

    public void empty(String tag) {
        empty(tag, getDefaultAttributes(tag));
    }

    public void empty(String tag, String attributes) {
	write("<");
	write(tag);
	if (attributes != null) {
	    write(" ");
	    write(attributes);
	}
	write("/>");
    }

    public void empty(String tag, String[][] attributes) {
	write("<");
	write(tag);
        writeAttributes(attributes);
	write("/>");
    }

    public void select(String name, Enum[] values, Enum selected) {
        tag("select", "name=" + Strings.quote(name));
        newLine();
        for (Enum v: values) {
            tagged("option", 
                   "value=" + Strings.quote(v.name()) +
                       (v == selected ? " selected=\"selected\"" : ""),
                   Strings.capitalize(v.name().toLowerCase()));
            newLine();
        }
        end("select");
    }

}

// Issues:
// * In a more spohisticated version, the attribute defaults could
//   be treated per-attribute rather than all-or-noting.
// * Instead of one set of attribute defaults there could be "styles".
//   However, something similar can be done by using multiple
//   HtmlWriters wrapped around the same Writer.
