/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Mar 12 03:42:13 2006 by Jeff Dalton
 * Copyright: (c) 2005 - 2006, AIAI, University of Edinburgh
 */

package ix.iface.util;

import javax.swing.*;

import javax.swing.text.html.HTMLEditorKit;

import javax.swing.text.StyleConstants;

import javax.swing.text.Document;
import javax.swing.text.AbstractDocument;
import javax.swing.text.DocumentFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.AttributeSet;

import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;

import java.util.*;

import ix.util.*;

/**
 * An editable JEditorPane that lets the user edit only within "td" cells.
 *
 * <p>N.B. Use {@link #setDocText(String)} rather than
 * directly calling the <code>setText(String)</code> method.</p>
 */
public class HtmlTableEditorPane extends JEditorPane {

    protected HTMLEditorKit tableEditorKit = new TableEditorKit();

    protected boolean settingText = false; // true to deactivate the filter

    public HtmlTableEditorPane() {
	// If we didn't want to use our own editor-kit, we could just do
	// super("text/html", "");
	super();
	setEditorKitForContentType("text/html", tableEditorKit);
	setContentType("text/html");
	setDocText("");
	Debug.expectSame(tableEditorKit, getEditorKit());

	// When the pane is editable, hyperlinks won't work.
	setEditable(true);

    }

    public void setDocText(String text) {
	// /\/: For some reason, we can't get away with overriding
	// the JEditorPane setText(String) method and so have to
	// use a different name.
	// /\/: The javadoc for the JEditorPane setText(String)
	// method seems to say it's better to create a new model
	// (ie a new document) rather than just set the text, because
	//    1. Leaving the existing model in place means that
	//       the old view will be torn down, and a new view
	//       created, where replacing the document would avoid
	//       the tear down of the old view.
	//    2. Some formats (such as HTML) can install things into
	//       the document that can influence future contents. HTML
	//       can have style information embedded that would influence
	//       the next content installed unexpectedly. 
	try {
	    settingText = true;
	    setDocument(tableEditorKit.createDefaultDocument()); // /\/
	    setText(text);
	    Debug.expectSame(tableEditorKit, getEditorKit());
	}
	finally {
	    settingText = false;
	}
    }

    /**
     * Called after an element is edited if {@link #settingText}
     * is false -- ie if we're not in a call to {@link #setDocText(String)}.
     */
    protected void editedElement(Element elt) {
	;
    }

    public void describeHtmlDocument() {
	new HtmlDescriber(System.out)
	    .describe((HTMLDocument)getDocument());
    }

    class TableEditorKit extends IXHtmlEditorKit {

	TableEditorKit() { }

	public Document createDefaultDocument() {
	    AbstractDocument doc =
		(AbstractDocument)super.createDefaultDocument();
	    doc.setDocumentFilter(new TableDocumentFilter());
	    return doc;
	}

    }

    class TableDocumentFilter extends DocumentFilter {

	TableDocumentFilter() { }

	Element editing = null;	// valid only if editIsAllowed

	public void remove(DocumentFilter.FilterBypass fb,
			   int offset,
			   int length)
	       throws BadLocationException {
	    if (!editIsAllowed(offset, length))
		throw new BadLocationException("No", offset);
	    else {
		super.remove(fb, offset, length);
		if (!settingText)
		    editedElement(editing);
	    }
	}

	public void replace(DocumentFilter.FilterBypass fb,
			    int offset,
			    int length,
			    String text,
			    AttributeSet attrs)
	       throws BadLocationException {
	    if (!editIsAllowed(offset, length))
		throw new BadLocationException("No", offset);
	    else {
		super.replace(fb, offset, length, text, attrs);
		if (!settingText)
		    editedElement(editing);
	    }
	}

	boolean editIsAllowed(int offset, int length) {
	    // If length == 0, we're not deleting anything,
	    // but we still have to make sure we're editing
	    // a TD rather than a TH.
//  	    Debug.noteln("Offset", offset);
//  	    Debug.noteln("length", length);
	    // /\/: Why should it be offset + length - 1?
	    // Suppose one element, e0, starts at offset 0 and
	    // is 10 characters long.  The next element, e1, will
	    // start at offset 11.  If we were replacing the
	    // contents of e0, we'll be given offset=0, length=10.
	    // "From" and "to" should both be e0, but if we
	    // looked up 11 for "to", we'd get e1.
	    // But ... that doesn't work for some reason,
	    // while using offset + length does.
	    Element from = getTableCell(offset);
	    Element to = getTableCell(offset + length);
//  	    Debug.noteln("From", from);
//  	    Debug.noteln("To", to);
	    editing = from;
	    return settingText
		|| (from != null && to != null && from == to);
	}

//  	int getElementLength(Element e) {
//  	    return e.getEndOffset() - e.getStartOffset();
//  	}

	Element getTableCell(int offset) {
	    Document doc = HtmlTableEditorPane.this.getDocument();
	    Element elt = doc.getDefaultRootElement();
	    while (!elt.isLeaf()) {
		Object name =
		    elt.getAttributes()
		          .getAttribute(StyleConstants.NameAttribute);
		if (name == HTML.Tag.TD)
		    return elt;
		elt = elt.getElement(elt.getElementIndex(offset));
	    }
	    return null;
	}

    }

}

