/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Nov  3 16:20:03 2004 by Jeff Dalton
 * Copyright: (c) 2003, 2004, AIAI, University of Edinburgh
 */

package ix.iface.plan;

import java.net.URL;
import java.io.*;
import java.util.*;

import ix.iface.domain.SyntaxException;
import ix.icore.domain.*;
import ix.icore.plan.Plan;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Reads .init files as {@link Plan}s.
 */
public class InitLoader {

    public InitLoader() { }

    public Plan readInit(URL url) throws IOException {
	InputStream in = url.openStream();
	String line1 = Util.readLine(in);
	in.close();
	if (Strings.beforeFirst("=", line1).indexOf('.') > 0)
	    return readInitOldSyntax(url.openStream());
	else
	    return readInitNewSyntax(url);
    }

    Plan readInitNewSyntax(URL url) throws IOException {
	// Each line is just pattern=value without needing
	// parens around the pattern.
	BufferedReader in = new BufferedReader(Util.openURLReader(url));
	ListOfPatternAssignment state = new LinkedListOfPatternAssignment();
	while (true) {
	    String line = in.readLine();
	    if (line == null)
		break;
	    line = line.trim();
	    if (!line.equals("") && !line.startsWith(";"))
		state.add(parseConstraint(line));
	}
	// Return plan.
	Plan plan = new Plan();
	if (!state.isEmpty())
	    plan.setWorldState(state);
	return plan;
    }

    PatternAssignment parseConstraint(String spec) {
	String[] parts = Strings.breakAtFirst("=", spec);
	String pat = parts[0].trim();
	String val = parts[1].trim();
	if (val.equals(""))
	    val = "true";
	LList pattern = PatternParser.parse(pat);
	Object value = parseValue(val);
	return new PatternAssignment(pattern, value);
    }

    Plan readInitOldSyntax(InputStream stream) throws IOException {
	Properties props = new Properties();
	props.load(stream);
	MultiMap objectToFields = new MultiHashMap() {
	     public Collection makeValueCollection(Object firstItem) {
		 Set s = new TreeSet();
		 s.add(firstItem);
		 return s;
	     }
	};
	// Pass 1 -- make map of object names to field names.
	// For each object name, the field names are in a SortedSet.
	// This is to make it easier to create the pattern-assignments
	// in a nice order.
	for (Enumeration e = props.propertyNames(); e.hasMoreElements();) {
	    // Entries are objectName.fieldName=value
	    String pname = (String)e.nextElement();
	    String[] parts = Strings.breakAtFirst(".", pname);
	    String object = parts[0];
	    String field = parts[1];
	    if (object.equals("") || field.equals(""))
		throw new SyntaxException
		    ("Missing object or field in " + pname);
	    objectToFields.addValue(object, field);
	}
	// Pass 2 -- create pattern-assignments
	Plan plan = new Plan();
	ListOfPatternAssignment state = new LinkedListOfPatternAssignment();
	Set objects = new TreeSet(objectToFields.keySet()); // sorted
	for (Iterator i = objects.iterator(); i.hasNext();) {
	    String object = (String)i.next();
	    Set fields = (Set)objectToFields.get(object);
	    if (fields == null)
		throw new SyntaxException("No fields for " + object);
	    for (Iterator j = fields.iterator(); j.hasNext();) {
		String field = (String)j.next();
		String value = props.getProperty(object + "." + field);
		Debug.expect(value != null);
		LList pattern = Lisp.list(Symbol.intern(field),
					  Symbol.intern(object));
		PatternAssignment pv =
		    new PatternAssignment(pattern, parseValue(value));
		state.add(pv);
	    }
	}
	// Return plan.
	if (!state.isEmpty())
	    plan.setWorldState(state);
	return plan;
    }

    Object parseValue(String v) {
	try {
	    // Avoid an exception in the common case of more than
	    // one item in the value string, e.g. "not available".
	    LList items = Lisp.elementsFromString(v);
	    if (items.size() == 1)
		return items.get(0);
	    else
		Debug.note("More than one object in " +
			   Strings.quote(v) + "; ");
	}
	catch (Exception e) {
	    Debug.noteException(e, false);
	}
	Debug.noteln("Treating as string:", Strings.quote(v));
	return v;
    }

}
