/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Mar  6 17:56:08 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.net.URL;
import java.io.*;
import java.util.*;

import ix.iface.domain.SyntaxException;
import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Reads .checklist files as {@link Domain}s.
 */
public class ChecklistParser {

    ListOfRefinement refinements = new LinkedListOfRefinement();
    Gensym.Generator nameGen = new Gensym.Generator();

    URL url;
    BufferedReader in;
    LineParser lp;

    /**
     * Constructs a ChecklistParser for a given URL.
     */
    public ChecklistParser(URL url) throws IOException {
	this.url = url;
	this.in = new BufferedReader(Util.openURLReader(url));
	this.lp = new LineParser(in);
    }

    /**
     * Constructs a domain by parsing the contents of the URL.
     */
    public Domain readDomain() throws IOException {
	while (parseRefinement()) {
	}
	Domain dom = new Domain();
	dom.setName(nameFromURL(url));
	if (!refinements.isEmpty())
	    dom.setRefinements(refinements);
	return dom;
    }

    private String nameFromURL(URL url) {
	return Strings.beforeFirst(".", Strings.afterLast("/", url.getPath()));
    }

    /**
     * Parses the next top-level refinement and any indented within it.
     */
    boolean parseRefinement() throws IOException {
	Line line;
	// Get the first non-empty line.
	do { line = lp.nextLine(); }
	while (line.isEmpty() && line != EOF);
	// If we reached the end of the file, we're done.
	if (line == EOF)
	    return false;
	// Create a refinement using the current line as
	// its expands-pattern.
	LList expandsPattern = LList.newLList(line.words);
	Refinement r = new Refinement(makeRefinementName(expandsPattern),
				      expandsPattern);
	// Add it now, before any nested ones.
	refinements.add(r);
	// Look at the next line.  It's indentation level tells
	// us what to do.
	Line next = lp.nextLine();
	if (next.indent <= line.indent) {
	    // The next line has the same indentation or less,
	    // so it's not part of this refinement.  Therefore,
	    // the refinement doesn't have any nodes, and we
	    // should put the line back and return.
	    lp.pushLine(next);
	    return true;
	}
	// The next line had greater indentation, so lines at
	// its indentation level specify nodes of our refinement
	// (as well as possibly being expands-patterns of nested
	// refinements).
	ListOfNodeSpec nodes = new LinkedListOfNodeSpec();
	int nodeIndent = next.indent;
	int nodeCount = 1;
	while (next.indent == nodeIndent) {
	    // A line at the right indentation level specifies a node.
	    LList nodePattern = LList.newLList(next.words);
	    nodes.add(new NodeSpec(nodeCount++, nodePattern));
	    // Now peek ahead one line to see if we stay at the
	    // same indentation level, increase, or decrease.
	    Line peek = lp.nextLine();	// get it
	    lp.pushLine(peek);		// put it back
	    if (peek.indent > nodeIndent) {
		// The peeked line is indented further, so
		// the current "next" line is an expands pattern
		// for a nested refinement as well as a node pattern
		// for the refinement we're building.
		// So put that line back and call ourself
		// recursively.
		lp.pushLine(next);
		boolean itWorked = parseRefinement();
		Debug.expect(itWorked,
			     "failed to make nested refinement for line",
			     next);
	    }
	    next = lp.nextLine();
	}
	if (!nodes.isEmpty())
	    r.setNodes(nodes);
	lp.pushLine(next);	// we've looked one beyond the nodes lines
	return true;
    }

    String makeRefinementName(List expandsPattern) {
	return nameGen.nextString(expandsPattern.get(0).toString());
    }

    /**
     * A line represented as an indentation count and a list of words.
     */
    static class Line {
	int indent;
	List words;
	Line(int indent, List words) {
	    this.indent = indent;
	    this.words = words;
	    Debug.noteln("Parsed", this);
	}
	boolean isEmpty() {
	    return words.isEmpty();
	}
	public String toString() {
	    return "Line[" + indent + " " + words + "]";
	}
    }

    static Line EOF = new Line(-1, Collections.EMPTY_LIST);

    /**
     * Line-at-a-time parser with multi-line backup.
     */
    static class LineParser {
	private BufferedReader in;
	private LinkedList pushedLines = new LinkedList();
	LineParser(BufferedReader in) {
	    this.in = in;
	}
	Line nextLine() throws IOException {
	    if (!pushedLines.isEmpty()) {
		return (Line)pushedLines.removeFirst();
	    }
	    String line = in.readLine();
	    if (line == null)
		return EOF;
	    int indent = 0;
	    List parts = Strings.breakAt(" ", line);
	    List words = new LinkedList();
	    for (Iterator i = parts.iterator(); i.hasNext();) {
		String w = (String)i.next();
		if (w.equals("")) {
		    if (words.isEmpty())
			indent++;
		}
		else
		    words.add(makeWord(w));
	    }
	    return new Line(indent, words);
	}
	void pushLine(Line back) {
	    pushedLines.addFirst(back);
	}
	private Object makeWord(String text) {
	    return Symbol.intern(text);
	}
    }

}
