/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 10 11:40:22 2008 by Jeff Dalton
 * Copyright: (c) 2003, 2004, 2008, AIAI, University of Edinburgh
 */

package ix.icore.plan.inspect;

import java.util.*;

import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.Lisp;

/**
 * Plan-inspection utility.
 */
public class PlanInspector {

    protected Plan plan;

    protected boolean isAutoRecursive = true;

    private Map taskItemToPlanItemMap;

    private WeakHashMap taskItemToPlanItemVisitorCache = new WeakHashMap();

    public PlanInspector(Plan plan) {
	this.plan = plan;
    }

    public PlanInspector setIsAutoRecursive(boolean v) {
	isAutoRecursive = v;
	if (v == false && taskItemToPlanItemMap == null)
	    taskItemToPlanItemMap = new HashMap();
	return this;		// so a further method can be called /\/
    }

    public Set getIssueIdSet() {
	return getIdSet(plan.getPlanIssues());
    }

    public Set getActivityIdSet() {
	return getIdSet(plan.getPlanNodes());
    }

    protected Set getIdSet(List rootPlanItems) {
	final Set result = new HashSet();
	walkTaskItems(rootPlanItems, new TaskItemVisitor() {
	    public void visit(TaskItem item, TaskItem parent, List children) {
		if (item.getId() != null)
		    result.add(item.getId());
		if (!isAutoRecursive)
		    walkTaskItemChildren(item, this);
	    }
	});
	return result;
    }

    /*
     * Walking TaskItems (Issues and Activities)
     */

    public void walkIssues(TaskItemVisitor v) {
	walkTaskItems(plan.getPlanIssues(), v);
    }

    public void walkActivities(TaskItemVisitor v) {
	walkTaskItems(plan.getPlanNodes(), v);
    }

    protected void walkTaskItems(List rootPlanItems, final TaskItemVisitor v) {
	walkPlanItems(rootPlanItems, null, planItemVisitor(v));
    }

    public void walkTaskItemChildren(TaskItem item, TaskItemVisitor v) {
	AbstractPlanItem pi = toPlanItem(item);
	walkPlanItems(getSubitems(pi), pi, planItemVisitor(v));
    }

    public AbstractPlanItem toPlanItem(TaskItem ti) {
	AbstractPlanItem pi =
	    (AbstractPlanItem)taskItemToPlanItemMap.get(ti);
	if (pi == null)
	    throw new ConsistencyException
		("Could not find plan-item for task-item", ti);
	return pi;
    }

    private PlanItemVisitor planItemVisitor(TaskItemVisitor tiv) {
	PlanItemVisitor piv =
	    (PlanItemVisitor)taskItemToPlanItemVisitorCache.get(tiv);
	if (piv == null) {
	    piv = new PIVisitorWrapper(tiv);
	    taskItemToPlanItemVisitorCache.put(tiv, piv);
	}
	return piv;
    }

    private class PIVisitorWrapper implements PlanItemVisitor {
	TaskItemVisitor tiVisitor;
	PIVisitorWrapper(TaskItemVisitor v) {
	    this.tiVisitor = v;
	}
	public void visit(AbstractPlanItem pi,
			  AbstractPlanItem parent,
			  List children) {
	    tiVisitor.visit(toTaskItem(pi),
			    parent == null ? null : toTaskItem(parent),
			    toTaskItems(children));
	}
    }

    private TaskItem toTaskItem(AbstractPlanItem pi) {
	TaskItem ti = pi.getItem();
	if (taskItemToPlanItemMap != null)
	    taskItemToPlanItemMap.put(ti, pi);
	return ti;
    }

    private List toTaskItems(List planItems) {
	return (List)Collect.map(new LinkedList(), planItems, new Function1() {
	    public Object funcall(Object a) {
		return toTaskItem((AbstractPlanItem)a);
	    }
	});
    }

    /*
     * Walking PlanItems
     */

    public void walkPlanNodes(PlanItemVisitor v) {
	walkPlanItems(plan.getPlanNodes(), null, v);
    }

    protected void walkPlanItems(List planItems, // children of parent
				 AbstractPlanItem parent,
				 PlanItemVisitor v) {
	for (Iterator i = Collect.iterator(planItems); i.hasNext();) {
	    AbstractPlanItem pi = (AbstractPlanItem)i.next();
	    List children = getSubitems(pi); // grandchildren of parent
	    v.visit(pi, parent, children);
	    if (isAutoRecursive)
		walkPlanItems(children, pi, v);
	}
    }

    protected List getSubitems(AbstractPlanItem pi) {
	// Always returns a List, never null.
	Name id = pi.getExpansion();
	if (id == null)
	    return Lisp.NIL;	// no expansion
	else
	    // Remember that an "expansion" might not have subnodes.
	    return Collect.ensureList(getRefinement(id).getSubitems());
    }

    protected AbstractRefinement getRefinement(Name id) {
	try {
	    return plan.getPlanIssueRefinement(id);
	}
	catch (IllegalArgumentException e1) {
	    try {
		return plan.getPlanRefinement(id);
	    }
	    catch (IllegalArgumentException e2) {
		throw new IllegalArgumentException
		    ("No Plan or Plan Issue Refinement named " + id);
	    }
	}
    }

    /*
     * Walking orderings
     */

    public void walkOrderings(OrderingVisitor v) {
	// This is simpler than some of the other walks, because
	// we don't need to pass parent and children to the visitor
	// and hence don't need to work recursively.  We can simply
	// iterate over the PlanRefinements.

	// Top-level orderings.
	walkOrderings(plan.getConstraints(), v);

	// Sibling orderings at lower levels.
	ListOfPlanRefinement refinements = plan.getPlanRefinements();
	if (refinements != null) {
	    for (PlanRefinement pr: refinements) {
		walkOrderings(pr.getConstraints(), v);
	    }
	}

    }

    protected void walkOrderings(ListOfConstrainer constraints,
				 OrderingVisitor v) {
	if (constraints != null) {
	    for (Constrainer c: constraints) {
		if (c instanceof Ordering)
		    v.visit((Ordering)c);
	    }
	}
    }

}
