/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 26 15:48:08 2008 by Jeff Dalton
 * Copyright: (c) 2001 - 2002, 2006, 2008, AIAI, University of Edinburgh
 */

package ix.icore;

import java.util.*;
import java.io.Serializable;

import ix.icore.log.HistoryEvent;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Something an agent might be asked to do or to handle.  Essentially
 * this class is just to hold what Issues and Activities have in common,
 * which at this level is almost everything.
 */
public abstract class TaskItem extends AbstractIXObject
    implements Serializable, Cloneable, Sendable, HasStatus {

    protected LList pattern;
    protected Status status = Status.BLANK;
    protected Priority priority = null;
    protected Name id;		// this item, w/in this agent
    protected Name senderId;	// source agent name, if source is external
    protected Name ref;		// ref to use in any reports back
    protected YesNo reportBack;	// set in forwarded copy

    protected String shortDescription; // set when pattern's set

    protected Set patternVars;	// set when pattern's set

    protected static final Symbol HISTORY = Symbol.intern("history");

    protected TaskItem() {
	// this.id = generateId();
    }

    protected TaskItem(LList pattern) {
	this();
	setPattern(pattern);
    }

    public LList getPattern() { return pattern; }

    public void setPattern(LList pattern) {
	this.pattern = pattern;
	this.shortDescription = PatternParser.unparse(pattern);
	this.patternVars = Variable.varsIn(pattern);
    }

    public Object getVerb() { return pattern.car(); }

    public LList getParameters() { return pattern.cdr(); }

    public String getShortDescription() { return shortDescription; }

    public Set getPatternVars() { return patternVars; }

    public Set getUnboundVars() {
	return Variable.unboundVarsIn(patternVars);
    }

    public boolean isGround() {
       return getUnboundVars().isEmpty();
    }

    public Name getId() { return id; }

    // /\/: Used to have no setId method - always set by constructor
    public void setId(Name id) {
	Debug.expect(this.id == null, "Resetting id");
	this.id = id;
    }

    public void ensureId() {
	if (id == null)
	    id = generateId();
    }

    public abstract Name generateId();

    public Status getStatus() { return status; }

    public void setStatus(Status status) { this.status = status; }

    public void computeStatus() { // because of HasStatus interface /\/
	throw new UnsupportedOperationException();
    }

    // For priority, we sometimes want to know whether a value
    // has been assigned or whether we're free to assign a value
    // from a parent item.  See AgendaItem.

    public Priority getPriority() {
	return priority == null ? Priority.NORMAL : priority;
    }

    public void setPriority(Priority priority) { this.priority = priority; }

    public boolean priorityWasSet() { return priority != null; }

    public Name getSenderId() { return senderId; }

    public void setSenderId(Name id) { this.senderId = id; }

    public Name getRef() { return ref; }

    public void setRef(Name ref) { this.ref = ref; }

    public YesNo getReportBack() { return reportBack; }

    public void setReportBack(YesNo rep) { this.reportBack = rep; }

    // /\/: Should use a typed list for the history.

    public List getHistory() {
	return (List)getAnnotation(HISTORY);
    }

    public void setHistory(List history) {
	// /\/: setAnnotation in AbstractAnnotatedObject should
	// remove an annotation that's set to null.
	if (history == null && getHistory() != null)
	    removeAnnotation(HISTORY);
	else
	    setAnnotation(HISTORY, history);
    }

    public void addHistoryEvent(HistoryEvent event) {
	// Fill in the date and time.
	if (event.getDate() == null)
	    // Should be set when event made? /\/
	    event.setDate(new Date());
	// /\/: Fill in agent name as well?

	// Add the event to the history
	List history = getHistory();
	if (history == null) {
	    history = new LinkedList();
	    setHistory(history);
	}
	history.add(event);
    }

    /**
     * Sends a copy of this item to another agent, optionally asking
     * the destination agent to send reports back.
     */
    public void forwardTo(Object destination, boolean reportBack) {
	Name ipcName = Name.valueOf(IXAgent.getAgent().getAgentIPCName());
	ensureId();
	try {
	    TaskItem copy = (TaskItem)clone();
	    // /\/: Make sure we don't include a short id in what
	    // we send.
	    if (!Gensym.usingUniquePrefix())
		copy.id = null;
	    // /\/: Remove Variables unless we get a better idea.
	    copy.setPattern((LList)Variable.removeVars(this.getPattern()));
	    copy.setSenderId(ipcName);
	    copy.setReportBack(YesNo.valueOf(reportBack));
	    copy.setRef(this.id);
	    IPC.sendObject(destination, copy);
	    if (Gensym.usingUniquePrefix()) {
		// Remember where forwarded-to.  N.B. This annotation
		// is not in the copy.
		this.setForwardedTo(Name.valueOf(destination));
	    }
	}
	catch (CloneNotSupportedException e) {
	    throw new Error("Can't clone " + this);
	}
    }

    public static Symbol S_FORWARDED_TO = Symbol.intern("forwarded-to");

    public Name getForwardedTo() {
	return (Name)getAnnotation(S_FORWARDED_TO);
    }

    public void setForwardedTo(Name toName) {
	setAnnotation(S_FORWARDED_TO, toName);
    }

    public Object clone() throws CloneNotSupportedException {
	TaskItem copy = (TaskItem)super.clone();
	// copy.id = generateId();
	return copy;
    }

}
