/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Sep 11 17:30:56 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, 2006, AIAI, University of Edinburgh
 */

package ix.icore;

import java.util.*;
import java.io.Serializable;

import ix.util.*;
import ix.util.lisp.Symbol;

import ix.icore.event.*;

/**
 * Provides an implementation of the {@link Annotated} interface
 * plus support for {@link ix.icore.event.AnnotationListener}s.
 */
public abstract class AbstractAnnotatedObject
       implements Annotated, Serializable {

    protected Annotations annotations;

    protected List annotationListeners = new LinkedList();

    public AbstractAnnotatedObject() { }

    // The Map isn't necessarily an Annotations instance,
    // because we want to allow some other possibilities
    // in subclasses; however there still has to be an
    // annotations field, because the usual ClassSyntax
    // wants that in addition to get- and set-methods
    // for the field to be visible.  /\/

    protected Map annMap() {
        return annotations;
    }

    protected void makeAnnMap() {
        Debug.expect(annotations == null);
        annotations = new Annotations();
    }

   public Annotations getAnnotations() {
        Map a = annMap();
        return a == null ? null
            :  a.isEmpty() ? null
            // Remember that in a subclass the Map might not be an Annotations
            :  a instanceof Annotations ? (Annotations)a
            :  new Annotations(a);
    }

    public void setAnnotations(Annotations annotations) {
	this.annotations = annotations;
    }

    public Object getAnnotation(Object key) {
	Map m = annMap();
	if (m != null)
	    return m.get(key);
	else
	    return null;
    }

    public void setAnnotation(Object key, Object value) {
	Map m = annMap();
	if (m == null) {
	    makeAnnMap();
	    m = annMap();
	}
	m.put(key, value);
	fireSetAnnotation(key, value);
    }

    public void removeAnnotation(Object key) {
	Object value = getAnnotation(key);
	Debug.expect(value != null, "Deleting valueless key", key);
	annMap().remove(key);
	fireSetAnnotation(key, null);
    }

    public void clearAnnotations() {
	if (annMap() != null) annMap().clear();
    }

    /**
     * Calls {@link #setAnnotation(Object, Object)} on all
     * key-value pairs in the map.
     */
    public void takeAnnotations(Map map) {
	for (Iterator i = map.entrySet().iterator(); i.hasNext();) {
	    Map.Entry entry = (Map.Entry)i.next();
	    setAnnotation(entry.getKey(), entry.getValue());
	}
    }

    // Comments are a simple and common annotation, but eventually
    // they may be replaced by various more specific ones.

    private static final Symbol S_COMMENTS = Symbol.intern("comments");

    public String getComments() {
	return (String)getAnnotation(S_COMMENTS);
    }

    public void setComments(String comments) {
	if (comments != null)
	    setAnnotation(S_COMMENTS, comments);
	else if (getAnnotation(S_COMMENTS) != null)
	    removeAnnotation(S_COMMENTS);
    }

    public void addAnnotationListener(AnnotationListener listener) {
	annotationListeners.add(listener);
    }

    public void fireSetAnnotation(Object key, Object value) {
	if (!annotationListeners.isEmpty()) {
	    AnnotationEvent event =
		new AnnotationEvent(this, key, value);
	    for (Iterator i = annotationListeners.iterator(); i.hasNext();) {
		AnnotationListener listener = (AnnotationListener)i.next();
		listener.setAnnotation(event);
	    }
	}
    }

}
