/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Aug 23 14:48:32 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.applet;

import java.io.*;
import java.util.*;

import ix.util.*;
import ix.util.ipc.*;

/**
 * Passes messages between an {@link AppletCommunicationStrategy}
 * and a {@link MessageServer}.
 */
public class CGIRelay {

    ServiceAddress serverAddr =
	new ServiceAddress(Parameters.getParameter("server-address"));

    MessageServer.Strategy strategy =
	new MessageServer.Strategy(serverAddr);

    public CGIRelay() {
    }

    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);
	try {
	    new CGIRelay().handleRequest();
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    System.exit(1);
	}
	System.exit(0);
    }

    void handleRequest() throws Exception {
	// Read the content.  We leave it as a string, but the
	// string contains an AppletMessage encoded as XML.
	String input = getInput();

	// Now wrap it in (another) AppletMessage to include
	// the name and address of the machine that made the
	// request.
	AppletMessage message = new AppletMessage();
	message.setCommand(input);
	message.setRemoteHost(System.getProperty("cgi.remote_host"));
	message.setRemoteAddr(System.getProperty("cgi.remote_addr"));

	// See what the message server thinks of the request.
	// The reply is a string containing the XML representation
	// of an object.  If the message server caught an exception
	// while trying to handle the reqauest, and it could still
	// send a reply, then the reply will contain the XML for
	// a string that describes the exception.  That string should
	// begin with "Server exception: ".
	String reply;
	try {
	    reply = askMessageServer(message);
	}
	catch (Exception e) {
	    reply = "Exception: " + Debug.describeException(e);
	}

	// Return the message server's reply (a string of XML)
	// or else a string beginning "Exception: ".  Note that
	// the XML case might also be an exception description
	// -- see earlier comment in this method.
	try {
	    printReply(reply);
	    sendStatusToMessageServer("ok");
	}
	catch (Exception e) {
	    sendStatusToMessageServer
		("Exception: " + Debug.describeException(e));
	    throw e;
	}
    }

    String getInput() throws IOException, UnsupportedEncodingException {
	BufferedReader read =
	    new BufferedReader
	        (new InputStreamReader(System.in, "utf-8"));
	String line;
	List lines = new LinkedList();
	while ((line = read.readLine()) != null) {
	    lines.add(line);
	}
	return Strings.joinLines(lines);
    }

    String askMessageServer(AppletMessage message) {
	return (String)strategy.sendRequest("message-server", message);
    }

    void printReply(String reply) throws IOException,
                                         UnsupportedEncodingException  {
	// Unfortunately, Java PrintStreams, such as System.out, don't
	// throw IOExceptions.  Instead, we have to call checkError()
	// which tells us if there was an error but not what the error
	// was.
	System.out.println("");
	if (System.out.checkError()) // also flushes
	    throw new IPC.IPCException("Error before printing reply.");
	Writer send = new OutputStreamWriter(System.out, "utf-8");
	send.write(reply);
	send.flush();
	if (System.out.checkError())
	    throw new IPC.IPCException("Error after printing reply.");
	send.close();
    }

    void sendStatusToMessageServer(String status) {
	// The message server expects to get this from its
	// existing connection to this CGIRelay, set up by
	// our earlier call to askMessageServer.
	strategy.sendObject("message-server", status);
    }

}
