/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 25 15:00:04 2010 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.iplan;

import org.mindswap.owls.*;
import org.mindswap.owls.io.*;
import org.mindswap.owls.service.*;
import org.mindswap.owls.process.*;

import com.hp.hpl.jena.rdf.model.*;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.iplan.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.process.*;
import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;

import ix.util.owls.*;

/**
 * An I-X planner that can do some OWL-S input and output.
 */
public class PlanWithOWLS extends Slip {

    // Command-line arguments used by O-Plan CGI web demos:
    //   service-list=URI     -- OWL-S definitions to load into domain.
    //   owls-out=filename    -- where to write plan as OWL-S when done.
    //   domain-out=filename  -- where to write the Domain once read.
    //
    // We also use one defined in PlannerBase:
    //   output=filename      -- where to write the Plan when done.
    //
    // N.B. The service-list URI must begin "file:" in order
    // to refer to a file.  Otherwise, there'll be an exception
    // complaining that there's no scheme in the URI.
    //
    // If the service-list URI's path doesn't end in "owls",
    // it is loaded directly as a domain.

    public PlanWithOWLS(boolean standAlone) {
	super(standAlone);
    }

    public static void main(String[] argv) {
	Util.printGreeting("I-Plan with OWL-S");
	PlanWithOWLS p = new PlanWithOWLS(true);
	p.mainStartup(argv);
	p.plan();
	p.outputPlan();
	p.outputOwlSPlan();
	p.exit();
    }

    protected void readDomain(Domain domain) {
	// Start with any Domain specified by the "domain" parameter.
	super.readDomain(domain);
	// If the "service-list" parameter has a value, translate
	// any service descriptions found there and add them as
	// Refinements to the domain.
	String servicesURI = Parameters.getParameter("service-list");
	if (servicesURI != null) {
	    if (servicesURI.endsWith(".owls")) {
		ServiceTranslator t = new ServiceTranslator();
		Domain d = t.readServiceDomain(servicesURI);
		domain.takeFrom(d);
	    }
	    else {
		Domain d = (Domain)XML.readObject(Domain.class, servicesURI);
		domain.takeFrom(d);
	    }
	}
	// If the "domain-out" parameter has a value, write
	// the domain to the file it specifies.
	String domainOut = Parameters.getParameter("domain-out");
	if (domainOut != null) {
	    XML.writeObject(domain, domainOut);
	}
    }

    protected void outputOwlSPlan() {
	String whereTo = Parameters.getParameter("owls-out");
	if (whereTo == null)
	    return;
	PlanToOWLS owls = new PlanToOWLS(this);
	Service service = owls.planToService();
	try {
	    OutputStream out = 
		new BufferedOutputStream(new FileOutputStream(whereTo));
	    ProcessWriter writer = new ProcessWriter();
	    writer.write(service, out);
	    out.flush();
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    throw new RethrownException(e);
	}
    }

}
