/****************************************************************************
 * Static utilities for IView things
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Thu Jun 26 17:32:14 2008 by Jeff Dalton
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
/** Changes
 * 11/12: changed sortTICollection for Jeff
 */

package ix.iview.util;

import ix.iface.domain.SyntaxException;
import ix.iface.domain.LTF_Parser;
import ix.iface.ui.util.*;
import ix.iface.ui.tree.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iview.domain.*;
import ix.iview.tree.*;
import ix.iview.*;
import ix.util.*;
import ix.util.lisp.*;
import java.lang.reflect.*;
import java.util.*;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.net.URL;
import javax.swing.*;
import javax.swing.text.JTextComponent;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.*;


/** Class for useful static methods. */

public class IVUtil {

  //--------------------Release things-----------------------

  public static String ideReleaseToString() {
    return ix.iview.IDERelease.name 
      + " version " + ix.iview.IDERelease.version
      + ", " + ix.iview.IDERelease.date;
  }
  public static String releaseToString() {
    return "I-X version " + ix.Release.version
      + ", " + ix.Release.date;
  }
  public static void printIDEReleaseGreeting() {
    System.out.println(ideReleaseToString());
	System.out.println("");
  }


  //-------------------- String Searching -----------------------

  public static int findString(String pattern, String text){
    Pattern patt = Pattern.compile(pattern, Pattern.CASE_INSENSITIVE);

    Matcher matcher = patt.matcher(text);

    if (matcher.find()) {
	Debug.noteln("I found the text starting at ", matcher.start());
	return matcher.start();
    }
    else {
	Debug.noteln("No match found.");
	return -1;
    }
  }

  /**
   * Finds objects in the given list whose names match the given string. 
   * @return the matches sorted by how early the string appears in the name.
   */
  public static Set findNameMatches(String pattern, List namedThings) {
    if (namedThings == null) return null;
    Pattern patt = Pattern.compile(pattern, Pattern.CASE_INSENSITIVE);

    TreeMap matchThings = new TreeMap(); //start--objectList map for sorting
    for (Iterator i = namedThings.iterator(); i.hasNext(); ) {
      Named namedThing = (Named)i.next();
      //Debug.noteln(" matching", namedThing.getName());
      Matcher matcher = patt.matcher(namedThing.getName());
      if (matcher.find(0)) {
	Object p = matchThings.get(new Integer(matcher.start()));
	LinkedList previous;
	if (p == null) previous = new LinkedList();
	else previous = (LinkedList)p;
	previous.add(namedThing);
	matchThings.put(new Integer(matcher.start()), previous);
      }
      //else Debug.noteln("   no match");
    }
    if (matchThings.isEmpty()) return null;
    else {
      Collection matches = UIUtil.getFlatValues(matchThings);
      if ((matches == null) || matches.isEmpty()) return null;
      else return new LinkedHashSet(matches);

    }
  }


  //-------------------- Sorting -----------------------

  public static Collection sortNamedCollection(Collection c) {
    if (c == null) return null;
    Map sortMap = new TreeMap();
    Collection empties = new ArrayList(); //collect empty names seperately
    for (Iterator i = c.iterator(); i.hasNext(); ) {
      Named n = (Named)i.next();
      if (n != null) { //ignore null elements
	String name = n.getName();
	if ((name == null) || (name.equals(""))) empties.add(n);
	else sortMap.put(name, n);
      }
    }
    Collection sorted = new ArrayList(sortMap.values()); 
    sorted.addAll(empties);
    return sorted;
  }

  public static List sortNodeCollection(Collection c) {
    if (c == null) return null;
    try {
      TreeMap sortMap = new TreeMap();
      List unsorted = new ArrayList(); //remember things that cannot get sorted
      for (Iterator i = c.iterator(); i.hasNext(); ) {
	NodeSpec n = (NodeSpec)i.next();
	if (n != null) { //ignore null elements
	  Name id = n.getId();
	  //sortMap.put(id.toString(), n);
	  Object sorter;
	  try {
	    sorter = new Integer(id.toString()); //definitely sort integer ids
	    //sorter = new String(id.toString());
	    //sorter = id.toString();
	    sortMap.put(sorter, n);
	  }
	  catch (Exception e) { //not an integer id, so append later
	    unsorted.add(n);
	  }
	}
      }
      if (sortMap.values() == null) //nothing sorted, so just return
	return new ArrayList(c);
      else {
	List sorted = new ArrayList(sortMap.values()); 
	sorted.addAll(unsorted);
	return sorted;
      }
    }
    catch (Exception e2) { //if anything goes wrong, just don't sort
      Debug.noteln("Warning: sorting nodes failed - using un-sorted list",
		   UIUtil.show(c));
      return new ArrayList(c);
    }
  }  

  public static List sortConstraintCollection(Collection col) {
    if (col == null) return null;
    //Debug.noteln("IVU: sorting constraints - old", UIUtil.show(col));
    Map sortMap = new TreeMap();
    for (Iterator i = col.iterator(); i.hasNext(); ) {
      Constraint c = (Constraint)i.next();
      if (c != null) { //ignore null elements
	String typeRel = c.toString();
	if ((c.getType() != null) && c.getType().toString().equals("compute")) 
	  typeRel = "xxxx"+typeRel; //put compute constraints after world state
	sortMap.put(typeRel, c);
      }
      //else Debug.noteln("IVU: sorting constraints - got null constraint");
    }
    List sorted = new ArrayList(sortMap.values()); 
    //Debug.noteln("IVU: sorting constraints - new", UIUtil.show(sorted));
    return sorted;
  }  

  public static List sortTICollection(Collection c) {
    /*
    Map sortMap = new TreeMap();
    for (Iterator i = c.iterator(); i.hasNext(); ) {
      TaskItem ti = (TaskItem)i.next();
      if (ti != null) { //ignore null elements
	Name id = ti.getId();
	sortMap.put(id.toString(), ti);
      }
    }
    List sorted = new ArrayList(sortMap.values()); 
    return sorted;
    */

    // /\/: TaskItems no longer all have ids, and it's not clear
    // that anything should depend on ids reflecting creation order
    // in any case.  For now we'll just return them in their existing
    // order.  [JD Wed 27 Aug 03]
    return new ArrayList(c);
  }  


  //--------------------------- comparing things

  public static boolean sameList(List list, List otherList) {
    if (bothEmpty(list, otherList)) return true;
    else return ((list != null) && list.equals(otherList));
  }

  public static boolean sameMap(Map m1, Map m2) {
    if (bothEmpty(m1, m2)) return true;
    else return ((m1 != null) && m1.equals(m2));
  }
  public static boolean sameStableMap(Map m1, Map m2) {
    if (bothEmpty(m1, m2)) return true;
    else return ((m1 != null) && m1.equals(m2));
  }

  /* -- Gets applies to Annotations, which are no longer StableHashMaps.
     -- JD, 26 Jun 08.
  public static boolean sameStableMap(StableHashMap m1, StableHashMap m2) {
    if (bothEmpty(m1, m2)) return true;
    else return ((m1 != null) && m1.equals(m2));
  }
  */

  public static boolean sameSet(Collection list, Collection otherList) {
    if (bothEmpty(list, otherList)) return true;
    if ((list == null) || (otherList == null)) return false;
    if (list.isEmpty() || otherList.isEmpty()) return false;
    if (list.size() != otherList.size()) return false;
    ArrayList l = new ArrayList(list);
    for (Iterator i = l.iterator(); i.hasNext() ;) 
      if (!otherList.contains(i.next())) return false;
    return true;
  }
  public static boolean sameObject(Object thing, Object other) {
    if ((thing == null) && (other == null)) return true;
    if (thing == null) return false;
    else return thing.equals(other);
  }
  public static boolean sameString(String thing, String other) {
    if (((thing == null) || thing.equals("")) &&  
	((other == null) || other.equals(""))) return true;
    if (thing == null) return false; //above covers all same null
    else return thing.equals(other);
  }
  public static boolean sameSymbol(Object thing, Object other) {
    return sameString(thing.toString(), other.toString());
  }

  public static boolean bothEmpty(Collection list, Collection otherList) {
    if (((otherList == null) || (otherList.size() == 0)) &&
	((list == null) || (list.size() == 0)))
      return true;
    else return false;
  }
  public static boolean bothEmpty(Map map, Map otherMap) {
    return UIUtil.isEmptyThing(map) && UIUtil.isEmptyThing(otherMap);
    /*
    if (((otherMap == null) || otherMap.isEmpty() || 
	 isEmptyThing(otherMap.values()) &&
	((map == null) || map.isEmpty()))
      return true;
    else return false;
    */
  }






  //--------------------------- UIObject things

  /**
   * Check if the object should be overwritten or copied. Ask the user if 
   * copying makes sense.
   * @return true if the object is to be copied, false if not.
   */
  public static boolean checkCopyObject(String thing, UIObject uio, 
					String name) {
    if ((uio == null) || uio.isEmpty()) return false;
    else {
      String oldName = uio.getName();
      //Debug.noteln("IVU: Old name is", oldName);
      if ((oldName != null) && !oldName.equals("") && 
	  !oldName.equals("undefined") && !oldName.equals("Undefined") && 
	  (uio.nameHasChanged(name))) {
	String[] message = {"You have changed the name of the " + thing + ".",
			    "Do you want to overwrite the original",
			    "or define a new one?"};
	Object[] options = { "Overwrite", "New" };
	int y = JOptionPane.showOptionDialog(null, message, "Confirm", 
					     JOptionPane.DEFAULT_OPTION, 
					     JOptionPane.QUESTION_MESSAGE,
					     null, options, options[0]);
	Debug.noteln("IVUt: chose option", y);
	if (y == 1) return true; //chose "new", so do copy
      }
      return false;
    }
  }

  /**
   * Turn the given list of object classes into a list that shows the
   * tree (root first)
   */
  public static List makeObjectClassTree(UIDomain uid, List classList) {
    LinkedList classes = new LinkedList();
    //Debug.noteln("UID: getting object class tree from domain", print());
    ObjectClassTreeNode dummyRoot = 
      new ObjectClassTreeNode(new UIObjectClass(uid, "Dummy Root"));
    ObjectClassTreeModel classTreeModel = 
	new ObjectClassTreeModel(uid, dummyRoot);
    classTreeModel.clearModel();
    IXTrees.setupTreeModel(classTreeModel, classList);
    Object oRoot = classTreeModel.getRoot();
    if ((oRoot != null) && (oRoot instanceof ObjectClassTreeNode)) {
      ObjectClassTreeNode root = (ObjectClassTreeNode)oRoot;
      for (Enumeration e = root.depthFirstEnumeration(); 
	   e.hasMoreElements(); ) {
	Object node = e.nextElement(); //tree node
	//list wants root first
	classes.add(0,((ObjectClassTreeNode)node).getUserObject()); 
      }
    }
    //Debug.noteln("IVU:got object classes:", UIUtil.show(classes));
    return classes;
  }

  public static String getRelativeName(Object relative) {
    if (relative instanceof String) return (String)relative;
    else if (relative instanceof UIObjectClass) 
      return ((UIObjectClass)relative).getName();
    else if (relative instanceof ObjectClass) 
      return ((ObjectClass)relative).getName();
    else return null;
  }


  //--------------------------- Constraint things

  public static boolean isEmptyConstraint(Constraint constraint) {
    if (constraint == null) return true;
    List params = constraint.getParameters();
    if ((params == null) || params.isEmpty()) return true;
    else return false;
  }

  public static Constraint setConstraintValue(Constraint constr, Object val) {
    if (isEmptyConstraint(constr)) return constr;
    PatternAssignment pAss = constr.getPatternAssignment();
    if (pAss != null) pAss.setValue(val);
    return constr;
  }
  public static Constraint setConstraintValue(UIObjectClass oClass,
					      Constraint constr, Object val) {
    if (isEmptyConstraint(constr)) return constr;
    ObjectProperty prop = propertyFromConstraint(oClass, constr);
    if (prop != null) {
      PatternAssignment pAss = constr.getPatternAssignment();
      Object oldV = pAss.getValue();
      try {
	Symbol var = Symbol.intern(val.toString());
	if (var instanceof ItemVar) pAss.setValue(var);
	else {
	  ix.ip2.ObjectView.ValueParser vp = 
	    new ix.ip2.ObjectView.ValueParser();
	  Object newV = vp.read(prop, val.toString(), oldV);
	  pAss.setValue(newV);
	}
      }
      catch (Exception e) {
	Debug.noteException(e);
	pAss.setValue(val);
      }
    }
    return constr;
  }
  public static Object getConstraintValue(Constraint constr) {
    if (isEmptyConstraint(constr)) return null;
    PatternAssignment pAss = constr.getPatternAssignment();
    if (pAss != null) return pAss.getValue();
    return null;
  }

  public static String TYPE_STRING_a = "type";
  public static String TYPE_STRING_b = "class";
  /**
   * Looks through constraints and finds the most likely for class info.
   * (type ?var = Agent) or (class ?var = Agent) or (?var type = Agent)...
   * Looks for TYPE_STRING_a and first, then b (if a not found)
   * @return the reference to the object class as it appears in the assignment
   */
  public static Object findVariableClass(ItemVar var, List constraints) {
    if ((var == null) || (constraints == null)) return null;
    //try for type first
    List typeConstraints = 
      filterVariableProperty(var, TYPE_STRING_a, constraints);
    if (typeConstraints == null)
      typeConstraints = 
	filterVariableProperty(var, TYPE_STRING_b, constraints);
    if ((typeConstraints == null) || typeConstraints.isEmpty()) return null;
    else {
      Constraint c = (Constraint)typeConstraints.get(0);
      PatternAssignment ass = c.getPatternAssignment();
      Object value = ass.getValue();
      return value;
    }
  }

  public static Set removeConstraint(Set cSet, Constraint c)  {
    Set newSet = new HashSet();
    
    if ((cSet == null) || cSet.isEmpty()) return cSet;
    if (c == null) { 
      cSet.remove(c); 
      return cSet;
    }
    for (Iterator i = cSet.iterator(); i.hasNext(); ) {
      Object next = i.next();
      if (!(next instanceof Constraint))
	Debug.noteln(" found non-constraint in constraint set", next);
      else {
	if (!IVUtil.sameConstraint(c, (Constraint)next)) {
	  newSet.add(next);
	}
	//else Debug.noteln("Found c to remove!");
      }
    }
    cSet.clear();
    cSet.addAll(newSet);
    return cSet;
  }
  /**
   * Looks through constraints and finds all most likely class info.
   * (type ?var = Agent) or (class ?var = Agent) or (?var type =
   * Agent)...  Looks for TYPE_STRING_a and first, then b (if a not
   * found).  Leaves given list intact.
   */
  public static List filterClassConstraints(List constraints) {
    if (constraints == null) return null;
    //Debug.noteln("IVU: looking for " + TYPE_STRING_a);
    //Debug.noteln(" in ", UIUtil.show(constraints));
    //try for type first
    List typeConstraints = filterProperty(TYPE_STRING_a, constraints);
    if (typeConstraints == null)
      typeConstraints = filterProperty(TYPE_STRING_b, constraints);
    if ((typeConstraints == null) || typeConstraints.isEmpty()) return null;
    else {
      //Debug.noteln("  found it!");
      return typeConstraints;
    }
  }

  /**
   * Looks through constraints and finds the most likely for class info.
   * (type ?var = Agent) or (class ?var = Agent) or (?var type = Agent)...
   * Looks for TYPE_STRING_a and first, then b (if a not found)
   */
  public static boolean isClassConstraint(ItemVar var, Constraint c) {
    if ((var == null) || (c == null)) return false;

    if (isClassConstraint(c)) {
      //Debug.noteln("IVU: checking for class in", c);
      PatternAssignment ass = c.getPatternAssignment();
      if (ass == null) return false;
      LList pattern = ass.getPattern();
      if (pattern == null) return false;
      return isInPattern(var, pattern);
    }
    else return false;
  }
  /**
   * Looks through constraints and finds the most likely for class info.
   * (type ?var = Agent) or (class ?var = Agent) or (?var type = Agent)...
   * Looks for TYPE_STRING_a and first, then b (if a not found)
   */
  public static boolean isClassConstraint(Constraint c) {
    //Debug.noteln("IVU: checking for class in", c);
    //try for type first
    PatternAssignment ass = c.getPatternAssignment();
    if (ass == null) return false;
    LList pattern = ass.getPattern();
    if (pattern == null) return false;

    String aCap = UIUtil.toCapitalised(TYPE_STRING_a);
    String bCap = UIUtil.toCapitalised(TYPE_STRING_b);
    return (pattern.contains(Symbol.intern(TYPE_STRING_a)) || 
	    pattern.contains(Symbol.intern(aCap)) || 
	    pattern.contains(Symbol.intern(TYPE_STRING_b)) || 
	    pattern.contains(Symbol.intern(bCap)));
  }

  /**
   * Looks through constraints and finds the most likely for class info.
   * (type ?var = Agent) or (class ?var = Agent) or (?var type = Agent)...
   * Looks for TYPE_STRING_a and first, then b (if a not found)
   */
  public static Constraint findClassConstraint(ItemVar var, List constraints) {
    if ((var == null) || (constraints == null)) return null;
    //Debug.noteln("IVU: looking for ", var + ", " + TYPE_STRING_a);
    //Debug.noteln(" in ", UIUtil.show(constraints));
    //try for type first
    List typeConstraints = 
      filterVariableProperty(var, TYPE_STRING_a, constraints);
    if (typeConstraints == null)
      typeConstraints = 
	filterVariableProperty(var, TYPE_STRING_b, constraints);
    if ((typeConstraints == null) || typeConstraints.isEmpty()) return null;
    else {
      //Debug.noteln("  found it!");
      return (Constraint)typeConstraints.get(0);
    }
  }

  public static Object extractType(Constraint tConstraint) {
    PatternAssignment ass = tConstraint.getPatternAssignment();
    Object value = ass.getValue();
    return value;
  }
  public static Object extractType(Constraint tConstraint, UIDomain domain) {
    Object value = extractType(tConstraint);
    Object type = null;
    if (value instanceof String) 
      type = domain.getNamedObjectClass((String)value);
    if (type == null) type = value;
    return type;
  }


  /**
   * Looks at the pattern of the given constraint and returns the
   * string that is the first element of that pattern, i.e. the place
   * where a property would be.
   * @return the string in the first place of the pattern or "";
   */
  public static String propertyFromConstraint(Constraint c) {
    //Debug.noteln("IVU: prop from constraint", c);
    if (c == null) return "";
    PatternAssignment ass = c.getPatternAssignment();
    if (ass == null) return "";
    LList pattern = ass.getPattern();
    return getPropertyFromPattern(pattern);
  }
  /**
   * Looks at the pattern of the given constraint and returns the
   * string that is the first element of that pattern, i.e. the place
   * where a property would be.
   * @return the string in the first place of the pattern or "";
   */
  public static ObjectProperty propertyFromConstraint(UIObjectClass oClass,
						      Constraint c) {
    //Debug.noteln("IVU: prop from constraint", c);
    if (c == null) return null;
    PatternAssignment ass = c.getPatternAssignment();
    if (ass == null) return null;
    LList pattern = ass.getPattern();
    return propertyFromPattern(pattern, oClass);
  }
  /**
   * Looks for an ObjectProperty in the given pattern that is defined
   * for the given UIObjectClass. 
   * Can handle null values in both parameters.
   * return@ the ObjectProperty or null if none is found.
   */
  //can handle both null parameters
  public static ItemVar variableFromConstraint(Constraint c) {
    //Debug.noteln("IVU: var from constraint", c);
    if (c == null) return null;
    PatternAssignment ass = c.getPatternAssignment();
    if (ass == null) return null;
    LList pattern = ass.getPattern();
    String subj = getSubjectInPattern(pattern);
    if ((subj == null) || (subj.equals(""))) return null;
    Symbol sym = Symbol.intern(subj);
    if (sym instanceof ItemVar) return (ItemVar)sym;
    else return null;
  }
  /**
   * Looks for an ObjectProperty in the given pattern that is defined
   * for the given UIObjectClass. 
   * Can handle null values in both parameters.
   * return@ the ObjectProperty or null if none is found.
   */
  //can handle both null parameters
  public static ObjectProperty propertyFromPattern(List pattern, 
						   UIObjectClass oClass) {
    //Debug.noteln("IVU: prop from pattern for class", oClass);
    //Debug.noteln(" pattern:", UIUtil.show(pattern));
    if ((pattern == null) || (oClass == null)) return null;
    Collection properties = oClass.getAllProperties();
    //Debug.noteln(" properties:", UIUtil.show(properties));
    return propertyFromPattern(pattern, properties);
  }
  private static ObjectProperty propertyFromPattern(List pattern, 
						    Collection properties) {
    if (properties == null) return null;
    for (Iterator i = properties.iterator(); i.hasNext(); ) {
      ObjectProperty prop = (ObjectProperty)i.next();
      if (pattern.contains(prop.getName().toString()) || 
	  pattern.contains(prop.getName())) {
	//Debug.noteln(" found", prop);
	return prop;
      }
    }
    //Debug.noteln(" not found");
    return null;
  }
  
  /**
   * Finds a property of the given class whose name matches the given string.
   */
  public static ObjectProperty propertyFromString(String s, UIObjectClass oc) {
    if ((s == null) || (s.equals("")) || (oc == null)) return null;
    Collection props = oc.getAllProperties();
    for (Iterator i = props.iterator(); i.hasNext(); ) {
      ObjectProperty prop = (ObjectProperty)i.next();
      if (s.equals(prop.getName().toString())) return prop;
    }
    return null;  
  }

  /**
   * Looks through constraints and finds those for given property. Leaves given
   * list intact.
   */
  public static List filterProperty(String property, List constraints) {
    if ((constraints == null) || (property == "")) return null;
    List varPropConstraints = new ArrayList();
    String capProp = UIUtil.toCapitalised(property);
    Symbol capPropY = Symbol.intern(capProp);
    Symbol propertyY = Symbol.intern(property);
    for (Iterator i = constraints.iterator(); i.hasNext(); ) {
      try { 
	Constraint c = (Constraint)i.next();
	PatternAssignment ass = c.getPatternAssignment();
	if (ass != null) {
	  LList pattern = ass.getPattern();
	  if ((pattern != null) &&
	      (pattern.contains(propertyY) || pattern.contains(capPropY))) {
	    varPropConstraints.add(c);
	  }
	}
      } catch (Exception e) {} //only interested in constraints
    }
    return varPropConstraints;
  }

  /**
   * Looks through constraints and finds the most likely for class info.
   * (type ?var = Agent) or (class ?var = Agent) or (?var type = Agent)...
   * Looks for TYPE_STRING_a and b first, then i and j (if a and b not found)
   */
  public static List filterVariableProperty(ItemVar var, String property, 
					    List constraints) {
    if ((var == null) || (constraints == null) || (property == "")) 
      return null;
    List varPropConstraints = new ArrayList();
    String capProp = UIUtil.toCapitalised(property);
    Symbol capPropY = Symbol.intern(capProp);
    Symbol propertyY = Symbol.intern(property);
    for (Iterator i = constraints.iterator(); i.hasNext(); ) {
      try { 
	Constraint c = (Constraint)i.next();
	PatternAssignment ass = c.getPatternAssignment();
	if (ass != null) {
	  LList pattern = ass.getPattern();
	  if ((pattern != null) &&
	      (pattern.contains(propertyY) || pattern.contains(capPropY))) {
	    if (isInPattern(var, pattern))
	      varPropConstraints.add(c);
	  }
	}
      } catch (Exception e) {} //only interested in constraints
    }
    return varPropConstraints;
  }


  /**
   * Replace each variable in each given constraint with the new one
   * propertiesMap (property-constraintSet)
   */
  public static LList replaceInPattern(ItemVar oldVar, ItemVar newVar,
				       LList pattern) {
    if ((pattern == null) || pattern.isNull() || !isInPattern(oldVar, pattern))
      return pattern;
    if  ((oldVar == null) || (newVar == null)) return pattern;
    if (oldVar.equals(pattern.car()))
      return new Cons(newVar, replaceInPattern(oldVar, newVar, pattern.cdr()));
    else if (pattern.car() != null) {
      if (oldVar.toString().equals(pattern.car().toString())) 
	return new Cons(newVar, 
			replaceInPattern(oldVar, newVar, pattern.cdr()));
    }
    return new Cons(pattern.car(), 
		    replaceInPattern(oldVar, newVar, pattern.cdr()));
  }

  /**
   * Replace each variable in each given constraint with the new one
   * propertiesMap (property-constraintSet)
   */
  public static HashMap replaceVarInConstraints(ItemVar oldVar, ItemVar newVar,
						HashMap constraintsMap) {
    //Debug.noteln("IVU: replacing variable " +oldVar+ " with " +newVar+ " in",
    //		 constraintsMap);
    if (newVar == null) return new HashMap();
    if (newVar.equals(oldVar)) return constraintsMap;
    if (constraintsMap.values() == null) return constraintsMap;
    //Collection vals = new ArrayList(constraintsMap.values());
    for (Iterator i = constraintsMap.values().iterator(); i.hasNext(); ) {
      Set constraints = (Set)i.next();
      if (constraints != null) {
	for (Iterator ic = constraints.iterator(); ic.hasNext(); ) {
	  try { 
	    Constraint c = (Constraint)ic.next();
	    PatternAssignment ass = c.getPatternAssignment();
	    if (ass != null) {
	      LList newPatt = replaceInPattern(oldVar,newVar,ass.getPattern());
	      //Debug.noteln(" new pattern is", newPatt);
	      ass.setPattern(newPatt);
	    }
	  } catch (Exception e) { Debug.noteException(e); } 
	}
      }
    }
    //Debug.noteln(" new map:", constraintsMap);
    return constraintsMap;
  }

  public static boolean isInPattern(ItemVar var, LList pattern) {
    if (pattern == null) return false;
    if (var == null) return false;
    if (pattern.contains(var)) return true;
    if (pattern.contains(var.toString())) return true;
    String varName = var.toString();
    for (Iterator i = pattern.iterator(); i.hasNext(); ) {
      if (varName.equals(i.next().toString())) return true;
    }
    return false;
  }

  /**
   * Looks at the second element in the list and compares it to the given
   * variable.
   */
  public static boolean isSubjectInPattern(ItemVar var, LList pattern) {
    //check that pattern is at least 2 long: (prop subj ...)
    if ((pattern == null) || (pattern.size() < 2)) return false;
    if (var == null) return false;

    Object subj = pattern.get(1);
    if (subj == null) return false;
    return (var.equals(subj) || var.toString().equals(subj.toString()));
  }
  /**
   * Looks at the second element in the list returns it as a string.
   */
  public static String getSubjectInPattern(LList pattern) {
    //check that pattern is at least 2 long: (prop subj ...)
    if ((pattern == null) || (pattern.size() < 2)) return "";

    Object subj = pattern.get(1);
    return subj.toString();
  }
  /**
   * Looks at the first element in the list and compares it to the given
   * property.
   */
  public static boolean isPropertyPattern(ObjectProperty prop, LList pattern) {
    //check that pattern is at least 2 long: (prop subj ...)
    if ((pattern == null) || (pattern.size() < 1)) return false;
    if (prop == null) return false;

    Object pProp = pattern.get(0);
    if (pProp == null) return false;

    //Debug.noteln("IVUt: checking " + prop.toString() + " against", pProp);

    return (prop.equals(pProp) || 
	    prop.toString().equals(pProp.toString()) ||
	    prop.getName().toString().equals(pProp.toString()));
  }
  /**
   * Looks at the first element in the list and compares it to the given
   * property.
   */
  public static String getPropertyFromPattern(LList pattern) {
    //check that pattern is at least 2 long: (prop subj ...)
    if ((pattern == null) || (pattern.size() < 1)) return "";

    Object pProp = pattern.get(0);
    return pProp.toString();
  }

  /**
   * Looks at the second element in the list and compares it to the given
   * variable.
   */
  public static boolean isSimplePropertyAssignment(PatternAssignment ass,
						   ObjectProperty prop, 
						   ItemVar var) {
    LList pattern = ass.getPattern();
    return (isPropertyPattern(prop, pattern)
	    && isSubjectInPattern(var, pattern) 
	    && (pattern.size() == 2));
  }


  /**
   * Determines whether the two constraints are the same. Checks type,
   * relation, and the string version of parameters (printCOnstraintParameters)
   * Call this to compare constraints for the same property to work
   * out whether one should replace the other, or whether they are 
   */
  public static boolean sameConstraint(Constraint one, Constraint other) {
    if (one == null) {
      if (other == null) return true;
      else return false;
    }
    else if (other == null) return false;
    else if (!sameObjectConstraint(one, other)) return false;
    else {
      String paramsOne = printConstraintParameters(one);
      if (paramsOne != null) 
	return paramsOne.equals(printConstraintParameters(other));
      else return (printConstraintParameters(other) == null);
    }
  }
    
  /**
   * Determines whether the two constraints are on the same thing.
   * Call this to compare constraints for the same property to work
   * out whether one should replace the other, or whether they are 
   */
  public static boolean sameObjectConstraint(Constraint c, Constraint otherC) {
    return ((c.getType() == otherC.getType())
	    && (c.getRelation() == otherC.getRelation()));
  }

  public static Constraint readCondition(Symbol type, Symbol rel, String text) 
       throws SyntaxException {
    String[] parts = Strings.breakAtFirst("=", text);
    String pat = UIUtil.ensureParenthesized(parts[0].trim());
    String val = parts[1].trim();
    if (val.equals("")) val = "true";

    String parameters = "(" + type.toString() + " " + rel.toString() + " " + 
      pat + " = " + val + ")";
    Constraint c = null;
    try {
      List spec = (LList)Lisp.readFromString(parameters);
      LTF_Parser constraintParser = new LTF_Parser();
      c = constraintParser.parseConstraint(spec);
    }
    catch (Exception e) {
      throw new SyntaxException("Invalid constraint: " 
				+ Debug.foldException(e));
    }
    if (c == null) 
      throw new SyntaxException("Invalid constraint: " 
				+ type+"-"+rel+": "+text);	
    return c;
  }


  /**
   * This should be a method within Orderings and Constraints!
   */
  public static boolean checkNodeRefConsistency(UIRefinement uir, Ordering o) {
    if (o == null) return true;
    if (uir == null) return true;
    NodeEndRef nodeRef;
    NodeSpec node;
    nodeRef = o.getFrom();
    if (nodeRef == null) return false;
    else {
      node = uir.findNode(nodeRef.getNode());
      if (node == null) return false;
      else {
	nodeRef = o.getTo();
	if (nodeRef == null) return false;
	else {
	  node = uir.findNode(nodeRef.getNode());
	  if (node == null) return false;
	}
      }
    }
    return true; // no null or stale node references
  }


  //---------------------------- toText things------------------

  public static String namedListToDisplay(Collection l) {
    String text = "";
    if (l != null) {
      Iterator i = l.iterator();
      while (i.hasNext()) {
	try {
	  text = text + ((Named)i.next()).getName() + " ";
	}
	catch (Exception e) {
	  text = text + i.next().toString() + " ";
	}
      }
    }
    return text;
  }

  public static String textRenderPatterns(Collection specs) {
    if (specs == null) return "";
    LList pattern;
    String text = "";
    for (Iterator i = specs.iterator(); i.hasNext(); ) {
      Object item = i.next();
      if (item instanceof NodeSpec) 
	pattern = ((NodeSpec)item).getPattern();
      else if (item instanceof Issue) 
	pattern = ((TaskItem)item).getPattern();
      else {
	Debug.noteln("WARNING: Cannot get patterns from", item.getClass());
	pattern = Lisp.NIL;
      }
      text = text + Lisp.elementsToString(pattern) + UIUtil.lineSeparator;    
    }
    return text;
  }
  
  public String textDerenderPatterns(List specs) {
    if (specs == null) return "";
    LList pattern;
    String text = "";
    for (Iterator i = specs.iterator(); i.hasNext(); ) {
      Object item = i.next();
      if (item instanceof NodeSpec) 
	pattern = ((NodeSpec)item).getPattern();
      else if (item instanceof Issue) 
	pattern = ((TaskItem)item).getPattern();
      else {
	Debug.noteln("WARNING: Cannot get patterns from", item.getClass());
	pattern = Lisp.NIL;
      }
      text = text + Lisp.elementsToString(pattern) + UIUtil.lineSeparator;    
    }
    return text;
  }

  public static String printNodes(Collection nodes) {
    if (nodes == null) return "";
    String printed = "";
    for (Iterator i = nodes.iterator(); i.hasNext(); ) 
      printed = printed + ", " + printNode((NodeSpec)i.next());
    return printed;
  }
  public static String printOrderings(Collection orderings) {
    if (orderings == null) return "";
    String printed = "";
    for (Iterator i = orderings.iterator(); i.hasNext(); ) 
      printed = printed + ", " + printOrdering((Ordering)i.next());
    return printed;
  }
  public static String printVarDecs(Collection varDecs) {
    if (varDecs == null) return "";
    String printed = "";
    if (varDecs == null) printed = "any";
    else if (varDecs.size() == 0) printed = "none";
    else for (Iterator i = varDecs.iterator(); i.hasNext(); ) 
      printed = printed + ", " + printVarDec((VariableDeclaration)i.next());
    return printed;
  }

  public static String printNode(NodeSpec node) {
    return node.getPattern().toString();
  }
  public static String printVarDec(VariableDeclaration varDec) {
    return varDec.getName().toString();
  }
  public static String printOrdering(Ordering ordering) {
    if (ordering == null) return "";

    NodeEndRef fromE = ordering.getFrom();
    NodeEndRef toE = ordering.getTo();
    String f = "";
    String ft = "";
    String t = "";
    String tt = "";
    if ((fromE != null) && (fromE.getNode() != null))
      f = fromE.getNode().toString();
    if ((fromE != null) && (fromE.getEnd() != null))
      ft = fromE.getEnd().toString();
    if ((toE != null) && (toE.getNode() != null))
      t = toE.getNode().toString();
    if ((toE != null) && (toE.getEnd() != null))
      tt = toE.getEnd().toString();
    return f + "-" + ft + " -> " + t + "-" + tt;
  }

  public static String printCondition(Constraint cond) {
    if (cond == null) return "";

    //NodeEndRef fromE = cond.getAtNodeEnd();
    String condStr = "E: ";
    if (cond.getRelation() == Refinement.S_CONDITION) condStr = "C: ";

    String params = printConstraintParameters(cond);

    return condStr + params;
  }

  public static String printConstraints(Collection constraints) {
    if (constraints == null) return "";
    String printed = "";
    for (Iterator i = constraints.iterator(); i.hasNext(); ) {
      Object next = i.next();
      if (next instanceof Constraint) 
	printed = printed + ", " + printConstraint((Constraint)next);
      else Debug.noteln("IVUtil: this is not a constraint:", next);
    }
    return printed;
  }

  public static String printConstraint(Constraint constraint) {
    if (constraint == null) return "";

    //NodeEndRef fromE = cond.getAtNodeEnd();
    String type = "";
    try { type = constraint.getType().toString(); }
    catch (Exception e) {}
    String rel = "";
    try { rel = constraint.getRelation().toString(); }
    catch (Exception e2) {}

    String params = printConstraintParameters(constraint);

    return type + " - " + rel + ": " + params;
  }

  public static String printConstraintParameters(Constraint constraint) {
    if (constraint == null) return "";
    List params = constraint.getParameters();
    if (params == null) return "";
    String paramString = "";
    for (Iterator i = params.iterator(); i.hasNext(); ) {
      try {
	PatternAssignment pa = (PatternAssignment)i.next();
	paramString = paramString + 
	  pa.getPattern() + " = " + pa.getValue().toString() + " ";
      } catch (Exception e) {} //ignore things non-PatternAssignment
    }
    return paramString.trim();
  }
  public static String printUIObjects(Collection objects) {
    if (objects == null) return "";
    String printed = "";
    for (Iterator i = objects.iterator(); i.hasNext(); ) 
      printed = printed + ", " + UIUtil.lineSeparator + 
	  " " + ((UIObject)i.next()).print();
    return printed;
  }


  //--------------------------- Variable things

  public static void useVar(JTextComponent jtc, int position, String var) {
    String text = jtc.getText();
    String varText = "";
    if ((var == null) || (var == "")) varText = "";
    else if (var.equals("?")) varText = var;
    else varText = var + " ";
    String newText = text;
    if (position >= text.length()) //apparently, position may be after length!
      newText = text + varText;
    else {
      newText = text.substring(0, position) + varText;
      if (position + 1 <= text.length())
	newText = newText + text.substring(position, text.length());
    }
    jtc.setText(newText);
    //put caret after new variable or at end of text
    jtc.setCaretPosition(Math.min(newText.length(), 
				  position + varText.length()));
  }


  /**
   * Enables the use of declarations when a question-mark is typed in 
   * a text componenet.
   * NOTE: the "?" will not appear when typed!
   */
  public static void enableVars(VarSpecifier parent, JTextComponent jtc) {
    Action act = makeVarAction(parent, jtc);
    enableVarAction(jtc, act);
  }

  private static Action makeVarAction(final VarSpecifier parent, 
				      final JTextComponent jtc) {
    return new AbstractAction("startVar") {
      public void actionPerformed(ActionEvent ae) {
	int position = jtc.getCaretPosition();
	Point location = jtc.getCaret().getMagicCaretPosition();
	List varDecs = parent.getVarsToOffer();
	try {
	  offerVars(parent, jtc, varDecs, location, position);
	}
	catch (Exception ex) {
	  //offerVars(null, jtc, varDecs, location, position);
	}
      }
    };
  }

  private static void enableVarAction(JTextComponent jtc, Action act) {
    InputMap iMap = jtc.getInputMap(); //default: WHEN_FOCUSED
    ActionMap aMap = jtc.getActionMap();
    aMap.put("startVar", act);
    iMap.put(KeyStroke.getKeyStroke('?'), "startVar");
  }


  /**
   * Offers the declared variables (if any) for completion, splices in 
   * selection (if any)
   * NOTE: the "?" does not appear when typed!
   */
  public static void offerVars(final VarSpecifier parent, 
			       final JTextComponent jtc, 
			       List decs, Point location, final int position) {
    //Debug.noteln("IVU: offering vars");
    final Component comp;
    if (parent instanceof Component) comp = (Component)parent;
    else comp = null;

    if (decs == null) {//no construct or ANY
      //make sure the "?" appears
      useVar(jtc, position, "?");
      return;    
    }
    else if (decs.size() == 0) { //NONE
      String[] message = {"Declarations say that no variables are allowed",
			  "To change declarations use the Edit menu"};
      JOptionPane.showMessageDialog(comp, message);
      //removeVarStart(jtc, position);
    }
    else {//GIVEN
      //Debug.noteln("IVU: vars given", UIUtil.show(decs));
      //make a PopupMenu of the given ones and offer at location
      final JPopupMenu varPopup = new JPopupMenu();
      for (Iterator i = decs.iterator(); i.hasNext(); ) {
	VariableDeclaration dcl = (VariableDeclaration)i.next();
	final String var = dcl.getName().toString();
	varPopup.add(new AbstractAction(var) {
	  public void actionPerformed(ActionEvent e) {
	    useVar(jtc, position, var);
	    varPopup.setVisible(false);
	  }
	});
      }
      varPopup.addSeparator();
      varPopup.add(new AbstractAction("New...") {
	//let the user declare a new one (and insert it)
	public void actionPerformed(ActionEvent e) {
	  varPopup.setVisible(false);
	  String message = "Please enter the variable name";
	  String val = JOptionPane.showInputDialog(comp, message);
	  //declare the variable (may add a "?")
	  val = parent.addVariableDeclaration(val);
	  if (val != "") {
	    //insert it where we were.
	    useVar(jtc, position, val);
	    varPopup.setVisible(false);
	  }
	  jtc.validate();
	}
      });
      //Debug.noteln("IVU: showing var popup at", (int)location.getX()
      //		   + ", " + (int)location.getY());
      varPopup.show(jtc, (int)location.getX(), (int)location.getY());
    }
  }

  public static String ensureVarName(String name) {
    if (name == "") return "";
    else if (name.startsWith("?")) return name;
    else return "?" + name;
  }

  /**
   * Turns the given collection of variables (names or ItemVars) into a 
   * list of variable declarations
   */
  public static List makeVariableDeclarations(Collection vars) {
    if (vars == null) return null;
    ArrayList decs = new ArrayList();
    for (Iterator i = vars.iterator(); i.hasNext();) {
      Object o = i.next();
      if (o instanceof ItemVar) 
	decs.add(new VariableDeclaration((ItemVar)o));
      else if (o instanceof String) {
	VariableDeclaration var = makeVariableDeclaration((String)o);
	if (var != null) decs.add(var);
      }
      else {
	Debug.noteln("UIR: cannot deal with variable specifications of class",
		     o.getClass());
      }
    }
    return decs;
  }

  public static VariableDeclaration makeVariableDeclaration(String name) {
    String varName = ensureVarName(name);
    if (varName != "") {
      Symbol var = Symbol.intern(varName);
      if (var != null) 
	return new VariableDeclaration((ItemVar)var);
    }
    return null;
  }

  //This seems to be obsolete - the "?" does not appear when mapped to the 
  // action.
  //get rid of the "?"
  private static void removeVarStart(JTextComponent jtc, int position) {
    String text = jtc.getText();
    String newText = text.substring(0, position);
    if (position + 1 <= text.length())
      newText = newText + text.substring(position+1, text.length());
    jtc.setText(newText);
  }

}
